<?php
/**
 * AI Article Generator アップデート補助スクリプト
 * 
 * このスクリプトは設定を保持したままシステムを更新するために使用します。
 * 使用方法: ブラウザで /update.php にアクセス
 */

// エラー表示設定
error_reporting(E_ALL);
ini_set('display_errors', 1);

// 設定ファイルのパス
$configFile = __DIR__ . '/config.php';
$configBackupFile = __DIR__ . '/config.backup.php';
$configSampleFile = __DIR__ . '/config.sample.php';

// データファイルのパス
$dataDir = __DIR__ . '/data';
$presetsFile = $dataDir . '/presets.json';
$articlesFile = $dataDir . '/articles.json';
$historyFile = $dataDir . '/history.json';

// アクション処理
$action = $_POST['action'] ?? '';
$message = '';
$error = '';

if ($action === 'backup') {
    // 完全バックアップの作成
    $backupDir = __DIR__ . '/backup_' . date('Ymd_His');
    $backupFiles = [
        'config.php' => $configFile,
        'data/presets.json' => $presetsFile,
        'data/articles.json' => $articlesFile,
        'data/history.json' => $historyFile,
        'data/dmm_api_debug.log' => $dataDir . '/dmm_api_debug.log',
        'data/dmm_api_error.log' => $dataDir . '/dmm_api_error.log'
    ];
    
    if (!mkdir($backupDir, 0755, true)) {
        $error = 'バックアップディレクトリの作成に失敗しました。';
    } else {
        $backedUp = [];
        foreach ($backupFiles as $name => $path) {
            if (file_exists($path)) {
                $destPath = $backupDir . '/' . $name;
                $destDir = dirname($destPath);
                if (!is_dir($destDir)) {
                    mkdir($destDir, 0755, true);
                }
                if (copy($path, $destPath)) {
                    $backedUp[] = $name;
                }
            }
        }
        
        // config.phpもバックアップ（互換性のため）
        if (file_exists($configFile)) {
            copy($configFile, $configBackupFile);
        }
        
        $message = 'バックアップを作成しました: ' . implode(', ', $backedUp);
    }
} elseif ($action === 'restore') {
    // 設定の復元
    if (file_exists($configBackupFile)) {
        if (copy($configBackupFile, $configFile)) {
            $message = 'config.php を復元しました。';
        } else {
            $error = '復元に失敗しました。';
        }
    } else {
        $error = 'バックアップファイルが見つかりません。';
    }
} elseif ($action === 'backup_all') {
    // 全データのバックアップ
    $backupData = [
        'version' => '1.0.0',
        'timestamp' => date('Y-m-d H:i:s'),
        'config' => null,
        'presets' => null,
        'articles' => null,
        'history' => null
    ];
    
    // config.phpの内容を読み込み
    if (file_exists($configFile)) {
        $config = require $configFile;
        $backupData['config'] = [
            'openai_api_key' => $config['openai_api_key'] ?? '',
            'anthropic_api_key' => $config['anthropic_api_key'] ?? '',
            'google_api_key' => $config['google_api_key'] ?? '',
            'dmm_api_id' => $config['dmm_api_id'] ?? '',
            'dmm_affiliate_id' => $config['dmm_affiliate_id'] ?? '',
            'venice_api_key' => $config['venice_api_key'] ?? '',
            'perplexity_api_key' => $config['perplexity_api_key'] ?? '',
            'wordpress' => $config['wordpress'] ?? []
        ];
    }
    
    // プリセットデータ
    if (file_exists($presetsFile)) {
        $backupData['presets'] = json_decode(file_get_contents($presetsFile), true);
    }
    
    // 記事データ（最新10件のみ）
    if (file_exists($articlesFile)) {
        $articles = json_decode(file_get_contents($articlesFile), true) ?: [];
        $backupData['articles'] = array_slice($articles, 0, 10);
    }
    
    // 履歴データ（最新20件のみ）
    if (file_exists($historyFile)) {
        $history = json_decode(file_get_contents($historyFile), true) ?: [];
        $backupData['history'] = array_slice($history, 0, 20);
    }
    
    // バックアップをダウンロード
    $backupFilename = 'backup_' . date('Ymd_His') . '.json';
    header('Content-Type: application/json');
    header('Content-Disposition: attachment; filename="' . $backupFilename . '"');
    header('Cache-Control: no-cache, no-store, must-revalidate');
    echo json_encode($backupData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
} elseif ($action === 'restore_all' && isset($_FILES['restore_file'])) {
    // 全データの復元
    $uploadedFile = $_FILES['restore_file'];
    
    if ($uploadedFile['error'] === UPLOAD_ERR_OK) {
        $jsonContent = file_get_contents($uploadedFile['tmp_name']);
        $backupData = json_decode($jsonContent, true);
        
        if ($backupData !== null && isset($backupData['version'])) {
            $restored = [];
            
            // config.phpの復元
            if (isset($backupData['config']) && file_exists($configFile)) {
                $config = require $configFile;
                foreach ($backupData['config'] as $key => $value) {
                    if (isset($config[$key])) {
                        $config[$key] = $value;
                    }
                }
                
                $configContent = "<?php\n// 直接アクセスを防止\nif (basename(\$_SERVER['PHP_SELF']) == basename(__FILE__)) {\n    die('Direct access not permitted');\n}\n\nreturn " . var_export($config, true) . ";\n?>";
                
                if (file_put_contents($configFile, $configContent)) {
                    $restored[] = 'APIキー設定';
                }
            }
            
            // プリセットの復元
            if (isset($backupData['presets'])) {
                if (!is_dir($dataDir)) {
                    mkdir($dataDir, 0755, true);
                }
                if (file_put_contents($presetsFile, json_encode($backupData['presets'], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
                    $restored[] = 'プリセット';
                }
            }
            
            // 記事データの復元（オプション）
            if (isset($_POST['restore_articles']) && isset($backupData['articles'])) {
                if (file_put_contents($articlesFile, json_encode($backupData['articles'], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
                    $restored[] = '記事データ';
                }
            }
            
            // 履歴データの復元（オプション）
            if (isset($_POST['restore_history']) && isset($backupData['history'])) {
                if (file_put_contents($historyFile, json_encode($backupData['history'], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
                    $restored[] = '履歴データ';
                }
            }
            
            if (!empty($restored)) {
                $message = '以下のデータを復元しました: ' . implode(', ', $restored);
            } else {
                $error = '復元するデータがありませんでした。';
            }
        } else {
            $error = '無効なバックアップファイルです。';
        }
    } else {
        $error = 'ファイルのアップロードに失敗しました。';
    }
} elseif ($action === 'export') {
    // 設定のエクスポート
    if (file_exists($configFile)) {
        $config = require $configFile;
        
        // APIキーのみを抽出
        $apiKeys = [
            'openai_api_key' => $config['openai_api_key'] ?? '',
            'anthropic_api_key' => $config['anthropic_api_key'] ?? '',
            'google_api_key' => $config['google_api_key'] ?? '',
            'dmm_api_id' => $config['dmm_api_id'] ?? '',
            'dmm_affiliate_id' => $config['dmm_affiliate_id'] ?? '',
            'venice_api_key' => $config['venice_api_key'] ?? '',
            'perplexity_api_key' => $config['perplexity_api_key'] ?? '',
            'wordpress' => $config['wordpress'] ?? []
        ];
        
        // ダウンロード
        header('Content-Type: application/json');
        header('Content-Disposition: attachment; filename="api_keys_backup_' . date('Ymd_His') . '.json"');
        echo json_encode($apiKeys, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        exit;
    }
} elseif ($action === 'import' && isset($_FILES['import_file'])) {
    // 設定のインポート
    $uploadedFile = $_FILES['import_file'];
    
    if ($uploadedFile['error'] === UPLOAD_ERR_OK) {
        $jsonContent = file_get_contents($uploadedFile['tmp_name']);
        $importedKeys = json_decode($jsonContent, true);
        
        if ($importedKeys !== null) {
            // 現在の設定を読み込み
            $config = require $configFile;
            
            // APIキーを更新
            foreach ($importedKeys as $key => $value) {
                if (isset($config[$key])) {
                    $config[$key] = $value;
                }
            }
            
            // 設定を保存
            $configContent = "<?php\n// 直接アクセスを防止\nif (basename(\$_SERVER['PHP_SELF']) == basename(__FILE__)) {\n    die('Direct access not permitted');\n}\n\nreturn " . var_export($config, true) . ";\n?>";
            
            if (file_put_contents($configFile, $configContent)) {
                $message = 'APIキーをインポートしました。';
            } else {
                $error = '設定の保存に失敗しました。';
            }
        } else {
            $error = '無効なJSONファイルです。';
        }
    } else {
        $error = 'ファイルのアップロードに失敗しました。';
    }
}

// 現在の状態を確認
$hasConfig = file_exists($configFile);
$hasBackup = file_exists($configBackupFile);
?>
<!DOCTYPE html>
<html lang="ja">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>AI Article Generator - アップデート補助ツール</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.1/font/bootstrap-icons.css" rel="stylesheet">
</head>
<body>
    <div class="container mt-5">
        <div class="row justify-content-center">
            <div class="col-md-8">
                <div class="card">
                    <div class="card-header bg-primary text-white">
                        <h3 class="mb-0"><i class="bi bi-arrow-clockwise"></i> アップデート補助ツール</h3>
                    </div>
                    <div class="card-body">
                        <?php if ($message): ?>
                            <div class="alert alert-success alert-dismissible fade show" role="alert">
                                <i class="bi bi-check-circle"></i> <?php echo htmlspecialchars($message); ?>
                                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            </div>
                        <?php endif; ?>
                        
                        <?php if ($error): ?>
                            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                                <i class="bi bi-exclamation-triangle"></i> <?php echo htmlspecialchars($error); ?>
                                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                            </div>
                        <?php endif; ?>
                        
                        <h4 class="mb-3">設定の管理</h4>
                        
                        <div class="row g-3">
                            <!-- バックアップ -->
                            <div class="col-md-6">
                                <div class="card h-100">
                                    <div class="card-body">
                                        <h5 class="card-title"><i class="bi bi-save"></i> バックアップ</h5>
                                        <p class="card-text">現在の設定をバックアップします。アップデート前に実行してください。</p>
                                        <form method="post">
                                            <input type="hidden" name="action" value="backup">
                                            <button type="submit" class="btn btn-primary" <?php echo !$hasConfig ? 'disabled' : ''; ?>>
                                                <i class="bi bi-download"></i> バックアップ作成
                                            </button>
                                        </form>
                                        <?php if ($hasBackup): ?>
                                            <small class="text-muted d-block mt-2">
                                                <i class="bi bi-check-circle text-success"></i> バックアップあり
                                            </small>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- 復元 -->
                            <div class="col-md-6">
                                <div class="card h-100">
                                    <div class="card-body">
                                        <h5 class="card-title"><i class="bi bi-arrow-counterclockwise"></i> 復元</h5>
                                        <p class="card-text">バックアップから設定を復元します。</p>
                                        <form method="post">
                                            <input type="hidden" name="action" value="restore">
                                            <button type="submit" class="btn btn-warning" <?php echo !$hasBackup ? 'disabled' : ''; ?>>
                                                <i class="bi bi-upload"></i> 設定を復元
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- エクスポート -->
                            <div class="col-md-6 mt-3">
                                <div class="card h-100">
                                    <div class="card-body">
                                        <h5 class="card-title"><i class="bi bi-box-arrow-up"></i> エクスポート</h5>
                                        <p class="card-text">APIキーをJSONファイルとしてダウンロードします。</p>
                                        <form method="post">
                                            <input type="hidden" name="action" value="export">
                                            <button type="submit" class="btn btn-info text-white" <?php echo !$hasConfig ? 'disabled' : ''; ?>>
                                                <i class="bi bi-file-earmark-code"></i> エクスポート
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- インポート -->
                            <div class="col-md-6 mt-3">
                                <div class="card h-100">
                                    <div class="card-body">
                                        <h5 class="card-title"><i class="bi bi-box-arrow-in-down"></i> インポート</h5>
                                        <p class="card-text">JSONファイルからAPIキーをインポートします。</p>
                                        <form method="post" enctype="multipart/form-data">
                                            <input type="hidden" name="action" value="import">
                                            <div class="mb-2">
                                                <input type="file" class="form-control form-control-sm" name="import_file" accept=".json" required>
                                            </div>
                                            <button type="submit" class="btn btn-success">
                                                <i class="bi bi-file-earmark-arrow-down"></i> インポート
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <hr class="my-4">
                        
                        <h4 class="mb-3">完全バックアップ（プリセット含む）</h4>
                        
                        <div class="row g-3">
                            <!-- 完全バックアップ -->
                            <div class="col-md-6">
                                <div class="card h-100 border-primary">
                                    <div class="card-body">
                                        <h5 class="card-title text-primary"><i class="bi bi-archive"></i> 完全バックアップ</h5>
                                        <p class="card-text">APIキー、プリセット、記事データなど全てをバックアップします。</p>
                                        <form method="post">
                                            <input type="hidden" name="action" value="backup_all">
                                            <button type="submit" class="btn btn-primary">
                                                <i class="bi bi-cloud-download"></i> 完全バックアップ作成
                                            </button>
                                        </form>
                                        <small class="text-muted d-block mt-2">
                                            含まれるデータ: APIキー、プリセット、記事履歴
                                        </small>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- 完全復元 -->
                            <div class="col-md-6">
                                <div class="card h-100 border-success">
                                    <div class="card-body">
                                        <h5 class="card-title text-success"><i class="bi bi-cloud-upload"></i> 完全復元</h5>
                                        <p class="card-text">バックアップファイルから全データを復元します。</p>
                                        <form method="post" enctype="multipart/form-data">
                                            <input type="hidden" name="action" value="restore_all">
                                            <div class="mb-2">
                                                <input type="file" class="form-control form-control-sm" name="restore_file" accept=".json" required>
                                            </div>
                                            <div class="form-check mb-2">
                                                <input class="form-check-input" type="checkbox" id="restore_articles" name="restore_articles">
                                                <label class="form-check-label" for="restore_articles">
                                                    記事データも復元
                                                </label>
                                            </div>
                                            <div class="form-check mb-2">
                                                <input class="form-check-input" type="checkbox" id="restore_history" name="restore_history">
                                                <label class="form-check-label" for="restore_history">
                                                    履歴データも復元
                                                </label>
                                            </div>
                                            <button type="submit" class="btn btn-success">
                                                <i class="bi bi-arrow-repeat"></i> 完全復元実行
                                            </button>
                                        </form>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <hr class="my-4">
                        
                        <h4 class="mb-3">アップデート手順</h4>
                        <ol>
                            <li>上記の「バックアップ作成」ボタンで設定をバックアップ</li>
                            <li>新しいバージョンのファイルで上書き（config.php以外）</li>
                            <li>問題が発生した場合は「設定を復元」で元に戻す</li>
                        </ol>
                        
                        <div class="alert alert-info mt-3">
                            <i class="bi bi-info-circle"></i> 
                            <strong>ヒント:</strong> エクスポート機能を使用すれば、複数の環境で同じAPIキーを使用できます。
                        </div>
                    </div>
                    <div class="card-footer text-center">
                        <a href="admin/" class="btn btn-secondary">
                            <i class="bi bi-house"></i> 管理画面に戻る
                        </a>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>