/**
 * Textarea Resize Fix
 * 管理画面全体でtextareaのリサイズ機能を有効化
 */
document.addEventListener('DOMContentLoaded', function() {
    
    // 全てのtextarea要素を取得
    function enableTextareaResize() {
        const textareas = document.querySelectorAll('textarea');
        
        textareas.forEach(textarea => {
            // disabled または readonly でない場合のみ
            if (!textarea.disabled && !textarea.readOnly) {
                // 強制的にリサイズを有効化
                textarea.style.resize = 'vertical';
                textarea.style.overflow = 'auto';
                
                // 最小・最大高さを設定
                if (!textarea.style.minHeight) {
                    textarea.style.minHeight = '80px';
                }
                
                // リサイズハンドルの表示を確実にする
                textarea.style.webkitAppearance = 'textarea';
                textarea.style.mozAppearance = 'textfield-multiline';
                
                // CSSで設定されている可能性のあるリサイズ無効化を解除
                textarea.classList.remove('no-resize');
                
                // data属性でリサイズが無効化されている場合も解除
                if (textarea.dataset.resize === 'none') {
                    delete textarea.dataset.resize;
                }
            }
        });
    }
    
    // 初回実行
    enableTextareaResize();
    
    // 動的に追加されるtextareaにも対応
    const observer = new MutationObserver(function(mutations) {
        mutations.forEach(function(mutation) {
            if (mutation.type === 'childList') {
                mutation.addedNodes.forEach(function(node) {
                    if (node.nodeType === 1) { // Element node
                        if (node.tagName === 'TEXTAREA') {
                            enableTextareaResize();
                        } else if (node.querySelectorAll) {
                            const textareas = node.querySelectorAll('textarea');
                            if (textareas.length > 0) {
                                enableTextareaResize();
                            }
                        }
                    }
                });
            }
        });
    });
    
    // body要素を監視
    observer.observe(document.body, {
        childList: true,
        subtree: true
    });
    
    // モーダルが開いた時にも実行
    $(document).on('shown.bs.modal', function() {
        enableTextareaResize();
    });
    
    // タブが切り替わった時にも実行
    $(document).on('shown.bs.tab', function() {
        enableTextareaResize();
    });
    
    // AJAXリクエスト完了後にも実行
    $(document).ajaxComplete(function() {
        setTimeout(enableTextareaResize, 100);
    });
    
    // ページ遷移時にも実行（SPAの場合）
    window.addEventListener('popstate', function() {
        setTimeout(enableTextareaResize, 100);
    });
    
    // リサイズイベント時の処理
    let resizeTimer;
    window.addEventListener('resize', function() {
        clearTimeout(resizeTimer);
        resizeTimer = setTimeout(function() {
            enableTextareaResize();
        }, 250);
    });
    
    // デバッグ用：全てのtextareaの状態を確認
    console.log('Textarea Resize Fix: 有効化完了');
    console.log('Textarea要素数:', document.querySelectorAll('textarea').length);
});

// グローバル関数として公開（デバッグ用）
window.enableTextareaResize = function() {
    const textareas = document.querySelectorAll('textarea');
    console.log('手動でtextareaリサイズを有効化:', textareas.length, '個の要素');
    textareas.forEach((textarea, index) => {
        console.log(`Textarea ${index + 1}:`, {
            id: textarea.id,
            name: textarea.name,
            resize: textarea.style.resize,
            disabled: textarea.disabled,
            readOnly: textarea.readOnly
        });
    });
};