// AI Article Generator - Admin Scripts

// グローバル変数（windowオブジェクトに定義して重複を防ぐ）
if (typeof window.currentArticle === 'undefined') {
    window.currentArticle = null;
}
if (typeof window.currentImages === 'undefined') {
    window.currentImages = [];
}

// DOM読み込み完了時
document.addEventListener('DOMContentLoaded', function() {
    // ツールチップの初期化
    const tooltips = document.querySelectorAll('[data-bs-toggle="tooltip"]');
    tooltips.forEach(tooltip => new bootstrap.Tooltip(tooltip));
    
    // サイドバートグル（モバイル）
    initSidebarToggle();
    
    // アラートの自動非表示
    initAutoHideAlerts();
    
    // フォームの初期化
    initForms();
});

// サイドバートグル
function initSidebarToggle() {
    // 既存のトグルボタンがあるか確認
    const existingBtn = document.querySelector('.sidebar-toggle-btn');
    if (existingBtn) {
        return;
    }
    
    const toggleBtn = document.createElement('button');
    toggleBtn.className = 'btn btn-primary d-md-none position-fixed sidebar-toggle-btn';
    toggleBtn.style.cssText = 'top: 1rem; left: 1rem; z-index: 1100;';
    toggleBtn.innerHTML = '<i class="bi bi-list"></i>';
    
    toggleBtn.addEventListener('click', function() {
        const sidebar = document.querySelector('.sidebar, .nyx-sidebar');
        if (sidebar) {
            sidebar.classList.toggle('show');
            sidebar.classList.toggle('active');
        }
    });
    
    if (window.innerWidth < 768) {
        document.body.appendChild(toggleBtn);
    }
}

// アラートの自動非表示
function initAutoHideAlerts() {
    const alerts = document.querySelectorAll('.alert-dismissible');
    alerts.forEach(alert => {
        setTimeout(() => {
            const bsAlert = new bootstrap.Alert(alert);
            bsAlert.close();
        }, 5000);
    });
}

// フォームの初期化
function initForms() {
    // ファイル入力のカスタマイズ
    const fileInputs = document.querySelectorAll('input[type="file"]');
    fileInputs.forEach(input => {
        input.addEventListener('change', function() {
            const fileName = this.files[0]?.name || 'ファイルを選択';
            const label = this.nextElementSibling;
            if (label && label.classList.contains('custom-file-label')) {
                label.textContent = fileName;
            }
        });
    });
}

// 記事生成関連の関数
async function generateArticle(formData) {
    try {
        const response = await fetch('../api/service.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                action: 'generate',
                ...formData
            })
        });
        
        if (!response.ok) {
            throw new Error('Network response was not ok');
        }
        
        const result = await response.json();
        
        if (result.success) {
            window.currentArticle = result.content;
            window.currentImages = result.images || [];
            
            // モデル設定をsessionStorageに保存
            const modelSettings = {
                model: formData.model,
                temperature: formData.temperature || 0.7,
                top_p: formData.top_p || 0.9
            };
            sessionStorage.setItem('lastModelSettings', JSON.stringify(modelSettings));
            
            showResultModal(result);
        } else {
            showError(result.error || '記事の生成に失敗しました');
        }
        
    } catch (error) {
        showError('通信エラーが発生しました: ' + error.message);
    }
}

// 結果モーダルの表示
function showResultModal(result) {
    // 既存のモーダルインスタンスをクリーンアップ
    const existingModal = bootstrap.Modal.getInstance(document.getElementById('resultModal'));
    if (existingModal) {
        existingModal.dispose();
    }
    
    const modalContent = document.getElementById('resultContent');
    if (!modalContent) return;
    
    let html = result.content;
    
    // 画像の挿入
    if (result.images && result.images.length > 0) {
        result.images.forEach(img => {
            const imgHtml = `
                <figure class="text-center my-4">
                    <img src="${img.url}" alt="${img.alt}" class="img-fluid rounded shadow">
                    <figcaption class="text-muted small mt-2">${img.alt}</figcaption>
                </figure>
            `;
            
            // 適切な位置に画像を挿入
            const headingRegex = new RegExp(`(<h[1-3][^>]*>${escapeRegExp(img.section)}</h[1-3]>)`, 'i');
            html = html.replace(headingRegex, `$1\n${imgHtml}`);
        });
    }
    
    modalContent.innerHTML = html;
    
    // SNS投稿ボタンは削除（サイドバーから利用）
    
    // メタデータの表示
    if (result.metadata) {
        const metaHtml = `
            <div class="alert alert-info mt-4">
                <small>
                    <strong>モデル:</strong> ${result.metadata.model} | 
                    <strong>プロバイダー:</strong> ${result.metadata.provider} | 
                    <strong>生成時刻:</strong> ${result.metadata.timestamp} | 
                    <strong>推定トークン数:</strong> ${result.metadata.tokens_used} | 
                    <strong>文字数:</strong> ${result.metadata.character_count || '不明'}文字
                </small>
            </div>
        `;
        modalContent.innerHTML += metaHtml;
        
        // 文字数警告がある場合は表示
        if (result.metadata.warning) {
            const warningHtml = `
                <div class="alert alert-warning mt-2">
                    <i class="bi bi-exclamation-triangle"></i> ${result.metadata.warning}
                </div>
            `;
            modalContent.innerHTML += warningHtml;
        }
    }
    
    const modalElement = document.getElementById('resultModal');
    const modal = new bootstrap.Modal(modalElement);
    
    // モーダルが完全に非表示になったらbackdropを確実に削除
    modalElement.addEventListener('hidden.bs.modal', function handleHidden() {
        // すべてのbackdropを削除
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
        // bodyのmodal-openクラスを削除
        document.body.classList.remove('modal-open');
        document.body.style.removeProperty('padding-right');
        // イベントリスナーを削除
        modalElement.removeEventListener('hidden.bs.modal', handleHidden);
    });
    
    modal.show();
}

// クリップボードにコピー
async function copyToClipboard() {
    if (!currentArticle) return;
    
    try {
        // 新しいClipboard APIを使用
        if (navigator.clipboard && navigator.clipboard.writeText) {
            await navigator.clipboard.writeText(currentArticle);
            showToast('クリップボードにコピーしました', 'success');
        } else {
            // フォールバック: 従来の方法
            const textarea = document.createElement('textarea');
            textarea.value = currentArticle;
            textarea.style.position = 'fixed';
            textarea.style.opacity = '0';
            document.body.appendChild(textarea);
            
            textarea.select();
            const result = document.execCommand('copy');
            document.body.removeChild(textarea);
            
            if (result) {
                showToast('クリップボードにコピーしました', 'success');
            } else {
                showError('コピーに失敗しました');
            }
        }
    } catch (error) {
        console.error('コピーエラー:', error);
        showError('コピーに失敗しました: ' + error.message);
    }
}

// 記事のダウンロード
function downloadArticle() {
    if (!currentArticle) return;
    
    const format = prompt('ダウンロード形式を選択してください (html/markdown/text):', 'markdown');
    if (!format) return;
    
    let content = currentArticle;
    let filename = 'article';
    let mimeType = 'text/plain';
    
    switch (format.toLowerCase()) {
        case 'html':
            content = wrapInHTML(content);
            filename += '.html';
            mimeType = 'text/html';
            break;
        case 'markdown':
            filename += '.md';
            mimeType = 'text/markdown';
            break;
        case 'text':
            content = stripTags(content);
            filename += '.txt';
            break;
    }
    
    const blob = new Blob([content], { type: mimeType });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    a.click();
    URL.revokeObjectURL(url);
    
    showToast('ダウンロードを開始しました', 'success');
}

// プリセットとして保存
function saveAsPreset() {
    const name = prompt('プリセット名を入力してください:');
    if (!name) return;
    
    // 現在のフォームデータを取得
    const presetData = {
        name: name,
        model: document.getElementById('model').value,
        temperature: parseFloat(document.getElementById('temperature').value),
        max_tokens: parseInt(document.getElementById('max_tokens').value),
        system_prompt: document.getElementById('system_prompt').value,
        enable_image: document.getElementById('enable_images').checked,
        image_model: document.getElementById('image_model').value,
        image_style: document.getElementById('image_style').value,
        image_prompt: document.getElementById('image_prompt').value,
        created_at: new Date().toISOString(),
        updated_at: new Date().toISOString()
    };
    
    // APIに送信
    fetch('../api/preset.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({
            action: 'save',
            preset: presetData
        })
    })
    .then(response => response.json())
    .then(result => {
        if (result.success) {
            showToast('プリセットを保存しました', 'success');
            // プリセット選択ボックスを更新
            location.reload();
        } else {
            showToast('保存に失敗しました: ' + result.error, 'danger');
        }
    })
    .catch(error => {
        showToast('エラーが発生しました: ' + error.message, 'danger');
    });
}

// ダッシュボードから記事を表示
function viewArticleFromDashboard(id) {
    viewArticle(id);
}

// SNS投稿を直接作成
function createSnsDirectly(id) {
    alert('SNS投稿作成は、サイドバーの「SNS投稿作成」メニューからご利用ください。');
}

// 履歴の表示
function viewArticle(id) {
    fetch('../api/service.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({
            action: 'get_history'
        })
    })
    .then(response => response.json())
    .then(result => {
        if (result.success && result.history) {
            const item = result.history.find(h => h.id === id);
            if (item) {
                currentArticle = item.content;
                currentImages = item.images || [];
                
                // モーダルのHTMLを作成
                let modalHtml = `
                    <div class="modal fade" id="historyModal" tabindex="-1">
                        <div class="modal-dialog modal-xl">
                            <div class="modal-content">
                                <div class="modal-header">
                                    <h5 class="modal-title">${item.title || '無題'}</h5>
                                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                                </div>
                                <div class="modal-body">
                                    <div class="mb-3">
                                        <small class="text-muted">
                                            <strong>モデル:</strong> ${item.model} | 
                                            <strong>作成日時:</strong> ${item.created_at}
                                            ${item.keywords ? ' | <strong>キーワード:</strong> ' + item.keywords : ''}
                                        </small>
                                    </div>
                                    <div id="historyContent">${item.content}</div>
                                </div>
                                <div class="modal-footer">
                                    <button type="button" class="btn btn-secondary" onclick="copyToClipboard()">
                                        <i class="bi bi-clipboard"></i> コピー
                                    </button>
                                    <button type="button" class="btn btn-primary" onclick="downloadArticle()">
                                        <i class="bi bi-download"></i> ダウンロード
                                    </button>
                                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">閉じる</button>
                                </div>
                            </div>
                        </div>
                    </div>
                `;
                
                // 既存のモーダルを削除
                const existingModal = document.getElementById('historyModal');
                if (existingModal) {
                    existingModal.remove();
                }
                
                // モーダルを追加
                document.body.insertAdjacentHTML('beforeend', modalHtml);
                
                // 画像を挿入
                if (item.images && item.images.length > 0) {
                    let content = item.content;
                    item.images.forEach(img => {
                        const imgHtml = `
                            <figure class="text-center my-4">
                                <img src="${img.url}" alt="${img.alt}" class="img-fluid rounded shadow">
                                <figcaption class="text-muted small mt-2">${img.alt}</figcaption>
                            </figure>
                        `;
                        
                        // 適切な位置に画像を挿入
                        const headingRegex = new RegExp(`(<h[1-3][^>]*>${escapeRegExp(img.section)}</h[1-3]>)`, 'i');
                        content = content.replace(headingRegex, `$1\n${imgHtml}`);
                    });
                    document.getElementById('historyContent').innerHTML = content;
                }
                
                // メタデータの表示
                if (item.metadata) {
                    const metaHtml = `
                        <div class="alert alert-info mt-4">
                            <small>
                                <strong>Temperature:</strong> ${item.metadata.temperature || 'N/A'} | 
                                <strong>最大トークン:</strong> ${item.metadata.max_tokens || 'N/A'} | 
                                <strong>出力形式:</strong> ${item.metadata.output_format || 'N/A'}
                            </small>
                        </div>
                    `;
                    document.getElementById('historyContent').innerHTML += metaHtml;
                }
                
                // モーダルを表示
                const modal = new bootstrap.Modal(document.getElementById('historyModal'));
                modal.show();
                
                // モーダルが閉じられたら削除
                document.getElementById('historyModal').addEventListener('hidden.bs.modal', function handleHistoryModalHidden() {
                    // すべてのbackdropを削除
                    document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
                    // bodyのmodal-openクラスを削除
                    document.body.classList.remove('modal-open');
                    document.body.style.removeProperty('padding-right');
                    // モーダルを削除
                    this.remove();
                }, { once: true });
            } else {
                showError('記事が見つかりません');
            }
        } else {
            showError('履歴の取得に失敗しました');
        }
    })
    .catch(error => {
        showError('履歴の読み込みに失敗しました: ' + error.message);
    });
}

// ユーティリティ関数
function showToast(message, type = 'info') {
    const toastHtml = `
        <div class="toast align-items-center text-white bg-${type} border-0" role="alert">
            <div class="d-flex">
                <div class="toast-body">${message}</div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
            </div>
        </div>
    `;
    
    let container = document.querySelector('.toast-container');
    if (!container) {
        container = document.createElement('div');
        container.className = 'toast-container position-fixed top-0 end-0 p-3';
        container.style.zIndex = '9999';
        document.body.appendChild(container);
    }
    
    container.insertAdjacentHTML('beforeend', toastHtml);
    const toastEl = container.lastElementChild;
    const toast = new bootstrap.Toast(toastEl);
    toast.show();
    
    toastEl.addEventListener('hidden.bs.toast', () => {
        toastEl.remove();
    });
}

function showError(message) {
    showToast(message, 'danger');
}

function escapeRegExp(string) {
    return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
}

function wrapInHTML(content) {
    return `<!DOCTYPE html>
<html lang="ja">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Generated Article</title>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            line-height: 1.6;
            color: #333;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
        }
        h1, h2, h3 { color: #2c3e50; }
        img { max-width: 100%; height: auto; }
        figure { margin: 2em 0; text-align: center; }
        figcaption { margin-top: 0.5em; color: #666; font-size: 0.9em; }
    </style>
</head>
<body>
    ${content}
</body>
</html>`;
}

function stripTags(html) {
    const tmp = document.createElement('div');
    tmp.innerHTML = html;
    return tmp.textContent || tmp.innerText || '';
}

// 画像ダウンロード機能
async function downloadImage(imageUrl, filename = 'image.jpg') {
    try {
        // Base64画像の場合は直接ダウンロード
        if (imageUrl.startsWith('data:image')) {
            const link = document.createElement('a');
            link.href = imageUrl;
            link.download = filename;
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
            showToast('画像をダウンロードしました', 'success');
            return;
        }
        
        // 外部URLの場合はサーバー経由でダウンロード
        const formData = new FormData();
        formData.append('action', 'download_single');
        formData.append('image_url', imageUrl);
        formData.append('filename', filename);
        
        const response = await fetch('../api/image-download.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                action: 'download_single',
                image_url: imageUrl,
                filename: filename
            })
        });
        
        if (!response.ok) {
            throw new Error('ダウンロードに失敗しました');
        }
        
        const blob = await response.blob();
        const url = URL.createObjectURL(blob);
        const link = document.createElement('a');
        link.href = url;
        link.download = filename;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        URL.revokeObjectURL(url);
        
        showToast('画像をダウンロードしました', 'success');
    } catch (error) {
        showError('画像のダウンロードに失敗しました: ' + error.message);
    }
}

// すべての画像をZIPでダウンロード
async function downloadAllImages(images, articleTitle = 'images') {
    if (!images || images.length === 0) {
        showError('ダウンロードする画像がありません');
        return;
    }
    
    try {
        showToast('画像をパッケージ化しています...', 'info');
        
        const response = await fetch('../api/image-download.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                action: 'download_batch',
                images: images,
                article_title: articleTitle
            })
        });
        
        if (!response.ok) {
            const errorData = await response.json();
            throw new Error(errorData.error || 'ダウンロードに失敗しました');
        }
        
        const blob = await response.blob();
        const url = URL.createObjectURL(blob);
        const link = document.createElement('a');
        link.href = url;
        link.download = `${articleTitle}_images.zip`;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        URL.revokeObjectURL(url);
        
        showToast('すべての画像をダウンロードしました', 'success');
    } catch (error) {
        showError('画像のダウンロードに失敗しました: ' + error.message);
    }
}

// SNS投稿関連の関数は sns-post.js に移動しました

// 画像プレビューモーダルの表示
function showImagePreview(imageUrl, alt = '') {
    const modalHtml = `
        <div class="modal fade" id="imagePreviewModal" tabindex="-1">
            <div class="modal-dialog modal-lg">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">画像プレビュー</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body text-center">
                        <img src="${imageUrl}" alt="${alt}" class="img-fluid">
                        <p class="text-muted mt-2">${alt}</p>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-primary" onclick="downloadImage('${imageUrl}', '${alt.replace(/'/g, "\\'")}.jpg')">
                            <i class="bi bi-download"></i> ダウンロード
                        </button>
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">閉じる</button>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    // 既存のモーダルを削除
    const existingModal = document.getElementById('imagePreviewModal');
    if (existingModal) {
        existingModal.remove();
    }
    
    // モーダルを追加
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    
    // モーダルを表示
    const modal = new bootstrap.Modal(document.getElementById('imagePreviewModal'));
    modal.show();
    
    // モーダルが閉じられたら削除
    document.getElementById('imagePreviewModal').addEventListener('hidden.bs.modal', function handleImageModalHidden() {
        // すべてのbackdropを削除
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
        // bodyのmodal-openクラスを削除
        document.body.classList.remove('modal-open');
        document.body.style.removeProperty('padding-right');
        // モーダルを削除
        this.remove();
    }, { once: true });
}

// モーダルのクリーンアップ関数
function cleanupModals() {
    // すべてのモーダルbackdropを削除
    document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
    // bodyのmodal-openクラスを削除
    document.body.classList.remove('modal-open');
    document.body.style.removeProperty('padding-right');
    document.body.style.removeProperty('overflow');
}

// グローバルエラーハンドラー（モーダル関連のエラーをキャッチ）
window.addEventListener('error', function(event) {
    if (event.message && event.message.includes('modal')) {
        console.error('Modal error detected:', event);
        cleanupModals();
    }
});

// Export functions for use in inline scripts
window.AI_Generator = {
    generateArticle,
    copyToClipboard,
    downloadArticle,
    saveAsPreset,
    viewArticle,
    showToast,
    showError,
    downloadImage,
    downloadAllImages,
    showImagePreview,
    cleanupModals
};

// グローバルスコープにも配置（互換性のため）
window.viewArticle = viewArticle;
window.copyToClipboard = copyToClipboard;
window.downloadArticle = downloadArticle;
window.saveAsPreset = saveAsPreset;
window.showToast = showToast;
window.showError = showError;
window.downloadImage = downloadImage;
window.downloadAllImages = downloadAllImages;
window.showImagePreview = showImagePreview;
window.cleanupModals = cleanupModals;