// API Error Handler Improvement
// This file improves error handling for API calls without affecting other functionality

(function() {
    'use strict';
    
    // Store original generateArticle function
    const originalGenerateArticle = window.generateArticle;
    
    // Enhanced generateArticle with better error handling
    window.generateArticle = async function(formData) {
        try {
            const response = await fetch('../api/service.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    action: 'generate',
                    ...formData
                })
            });
            
            // Check HTTP status first
            if (!response.ok) {
                let errorMessage = `HTTPエラー: ${response.status} ${response.statusText}`;
                
                // Try to get more details from response
                try {
                    const errorData = await response.json();
                    if (errorData.error) {
                        errorMessage = errorData.error;
                    }
                } catch (e) {
                    // If response is not JSON, try text
                    try {
                        const errorText = await response.text();
                        if (errorText) {
                            console.error('APIレスポンス:', errorText);
                            // Check for common PHP errors
                            if (errorText.includes('Fatal error') || errorText.includes('Parse error')) {
                                errorMessage = 'サーバーエラー: PHPエラーが発生しました。管理者に連絡してください。';
                            }
                        }
                    } catch (e2) {
                        console.error('エラーレスポンスの読み取りに失敗:', e2);
                    }
                }
                
                throw new Error(errorMessage);
            }
            
            // Parse JSON response
            let result;
            try {
                result = await response.json();
            } catch (e) {
                console.error('JSONパースエラー:', e);
                const responseText = await response.text();
                console.error('レスポンステキスト:', responseText);
                throw new Error('サーバーからの応答が正しくありません。JSONパースエラー。');
            }
            
            if (result.success) {
                window.currentArticle = result.content;
                window.currentImages = result.images || [];
                
                // モデル設定をsessionStorageに保存
                const modelSettings = {
                    model: formData.model,
                    temperature: formData.temperature || 0.7,
                    top_p: formData.top_p || 0.9
                };
                sessionStorage.setItem('lastModelSettings', JSON.stringify(modelSettings));
                
                showResultModal(result);
            } else {
                // API returned success: false
                let errorMessage = result.error || '記事の生成に失敗しました';
                console.error('API エラー:', result);
                
                // デバッグ情報がある場合は詳細を表示
                if (result.debug_info) {
                    console.error('デバッグ情報:', result.debug_info);
                    if (result.debug_info.hint) {
                        errorMessage += '\n' + result.debug_info.hint;
                    }
                    if (result.debug_info.api_key_status) {
                        errorMessage += '\n' + result.debug_info.api_key_status;
                    }
                }
                
                showError(errorMessage);
            }
            
        } catch (error) {
            console.error('generateArticle エラー:', error);
            
            // More specific error messages
            let errorMessage = '通信エラーが発生しました: ';
            
            if (error.name === 'NetworkError' || error.message.includes('fetch')) {
                errorMessage += 'ネットワークに接続できません。接続を確認してください。';
            } else if (error.message.includes('JSON')) {
                errorMessage += 'サーバーの応答形式が正しくありません。';
            } else if (error.message.includes('HTTPエラー')) {
                errorMessage = error.message;
            } else {
                errorMessage += error.message;
            }
            
            showError(errorMessage);
            
            // Log detailed error info for debugging
            console.error('詳細なエラー情報:', {
                message: error.message,
                stack: error.stack,
                formData: formData
            });
        }
    };
    
    // Enhanced error display function
    const originalShowError = window.showError;
    window.showError = function(message) {
        console.error('[エラー]', message);
        
        // Add timestamp to error
        const timestamp = new Date().toLocaleTimeString('ja-JP');
        const enhancedMessage = `[${timestamp}] ${message}`;
        
        // Call original showError if it exists
        if (originalShowError && typeof originalShowError === 'function') {
            originalShowError(enhancedMessage);
        } else {
            // Fallback error display
            const errorDiv = document.createElement('div');
            errorDiv.className = 'alert alert-danger alert-dismissible fade show';
            errorDiv.innerHTML = `
                <strong>エラー:</strong> ${enhancedMessage}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            `;
            
            // Find appropriate container
            const container = document.querySelector('.container-fluid') || document.body;
            container.insertBefore(errorDiv, container.firstChild);
            
            // Auto-hide after 10 seconds
            setTimeout(() => {
                errorDiv.remove();
            }, 10000);
        }
    };
    
    console.log('Error handler improvement loaded');
})();