// Comprehensive JavaScript Fixes for Nyx AI Studio

(function() {
    'use strict';
    
    // ===== モーダル管理の改善 =====
    
    // モーダル表示前のクリーンアップ関数
    window.prepareModal = function(modalId) {
        // 既存のbackdropを削除
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
        
        // body要素のクラスとスタイルをリセット
        document.body.classList.remove('modal-open');
        document.body.style.removeProperty('overflow');
        document.body.style.removeProperty('padding-right');
        
        // 既存のモーダルインスタンスを破棄
        const modalEl = document.getElementById(modalId);
        if (modalEl) {
            const existingModal = bootstrap.Modal.getInstance(modalEl);
            if (existingModal) {
                existingModal.dispose();
            }
        }
    };
    
    // モーダル表示の改善版
    window.showModalSafely = function(modalId, options = {}) {
        prepareModal(modalId);
        
        const modalEl = document.getElementById(modalId);
        if (!modalEl) return null;
        
        // デフォルトオプション
        const defaultOptions = {
            keyboard: true,
            backdrop: true,
            focus: true
        };
        
        const modalOptions = Object.assign({}, defaultOptions, options);
        const modal = new bootstrap.Modal(modalEl, modalOptions);
        
        // hidden.bs.modalイベントの処理を改善
        modalEl.addEventListener('hidden.bs.modal', function cleanup(e) {
            // このモーダルのイベントのみ処理
            if (e.target !== modalEl) return;
            
            setTimeout(() => {
                // backdropの削除
                document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
                
                // bodyの状態をチェックして修正
                const hasVisibleModal = document.querySelector('.modal.show');
                if (!hasVisibleModal) {
                    document.body.classList.remove('modal-open');
                    document.body.style.removeProperty('overflow');
                    document.body.style.removeProperty('padding-right');
                }
            }, 150);
            
            // イベントリスナーを削除
            modalEl.removeEventListener('hidden.bs.modal', cleanup);
        });
        
        modal.show();
        return modal;
    };
    
    // ===== フォーム送信の改善 =====
    
    // 二重送信防止
    document.addEventListener('submit', function(e) {
        const form = e.target;
        const submitBtn = form.querySelector('[type="submit"]');
        
        if (submitBtn && !submitBtn.disabled) {
            // ボタンを無効化
            submitBtn.disabled = true;
            
            // 元のテキストを保存
            const originalText = submitBtn.innerHTML;
            
            // スピナーを表示
            if (submitBtn.querySelector('.spinner-border') === null) {
                submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>' + originalText;
            }
            
            // エラー時の復元処理
            const resetButton = () => {
                submitBtn.disabled = false;
                submitBtn.innerHTML = originalText;
            };
            
            // 10秒後に自動的にボタンを復元（タイムアウト対策）
            setTimeout(resetButton, 10000);
        }
    });
    
    // ===== レスポンシブ対応の改善 =====
    
    // ビューポート幅の変更を監視
    let resizeTimer;
    window.addEventListener('resize', function() {
        clearTimeout(resizeTimer);
        resizeTimer = setTimeout(function() {
            // モバイルメニューの状態をリセット
            if (window.innerWidth > 768) {
                const sidebar = document.querySelector('.nyx-sidebar');
                const overlay = document.querySelector('.mobile-overlay');
                if (sidebar) sidebar.classList.remove('active');
                if (overlay) overlay.classList.remove('active');
            }
        }, 250);
    });
    
    // ===== スクロール性能の改善 =====
    
    // スムーススクロール
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function(e) {
            const targetId = this.getAttribute('href');
            if (targetId === '#') return;
            
            const target = document.querySelector(targetId);
            if (target) {
                e.preventDefault();
                target.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        });
    });
    
    // ===== エラーハンドリングの改善 =====
    
    // グローバルエラーハンドラー
    window.addEventListener('error', function(event) {
        console.error('Global error:', event.error);
        
        // モーダル関連のエラーの場合
        if (event.message && (event.message.includes('modal') || event.message.includes('backdrop'))) {
            console.warn('Modal error detected, cleaning up...');
            document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
            document.body.classList.remove('modal-open');
            document.body.style.removeProperty('overflow');
            document.body.style.removeProperty('padding-right');
        }
    });
    
    // ===== DOMContentLoaded時の初期化 =====
    
    document.addEventListener('DOMContentLoaded', function() {
        // 初期クリーンアップ
        document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
        document.body.classList.remove('modal-open');
        
        // ツールチップの初期化（エラーハンドリング付き）
        try {
            const tooltipTriggerList = document.querySelectorAll('[data-bs-toggle="tooltip"]');
            tooltipTriggerList.forEach(el => new bootstrap.Tooltip(el));
        } catch (e) {
            console.warn('Tooltip initialization failed:', e);
        }
        
        // フォーム要素のオートフォーカス
        const autofocusElement = document.querySelector('[autofocus]');
        if (autofocusElement && typeof autofocusElement.focus === 'function') {
            setTimeout(() => autofocusElement.focus(), 100);
        }
        
        // iOS Safari対策：フォーカス時のズーム防止
        if (/iPad|iPhone|iPod/.test(navigator.userAgent)) {
            document.querySelectorAll('input[type="text"], input[type="password"], textarea').forEach(el => {
                el.addEventListener('focus', function() {
                    this.style.fontSize = '16px';
                });
                el.addEventListener('blur', function() {
                    this.style.fontSize = '';
                });
            });
        }
    });
    
    // ===== ユーティリティ関数 =====
    
    // デバウンス関数
    window.debounce = function(func, wait, immediate) {
        let timeout;
        return function() {
            const context = this, args = arguments;
            const later = function() {
                timeout = null;
                if (!immediate) func.apply(context, args);
            };
            const callNow = immediate && !timeout;
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
            if (callNow) func.apply(context, args);
        };
    };
    
    // クリップボードコピーの改善
    window.copyToClipboardImproved = async function(text) {
        try {
            // 新しいClipboard API
            if (navigator.clipboard && navigator.clipboard.writeText) {
                await navigator.clipboard.writeText(text);
                return true;
            }
            
            // フォールバック方法
            const textarea = document.createElement('textarea');
            textarea.value = text;
            textarea.style.position = 'fixed';
            textarea.style.top = '-9999px';
            textarea.style.left = '-9999px';
            document.body.appendChild(textarea);
            
            textarea.select();
            textarea.setSelectionRange(0, textarea.value.length);
            
            const result = document.execCommand('copy');
            document.body.removeChild(textarea);
            
            return result;
        } catch (err) {
            console.error('Copy failed:', err);
            return false;
        }
    };
    
    // ===== 定期的なクリーンアップ =====
    
    // 5秒ごとにモーダルの状態をチェック
    setInterval(function() {
        const hasBackdrop = document.querySelector('.modal-backdrop');
        const hasVisibleModal = document.querySelector('.modal.show');
        const hasModalOpen = document.body.classList.contains('modal-open');
        
        // 不整合がある場合はクリーンアップ
        if ((hasBackdrop || hasModalOpen) && !hasVisibleModal) {
            console.log('Cleaning up stale modal state...');
            document.querySelectorAll('.modal-backdrop').forEach(backdrop => backdrop.remove());
            document.body.classList.remove('modal-open');
            document.body.style.removeProperty('overflow');
            document.body.style.removeProperty('padding-right');
        }
    }, 5000);
    
    // ===== パフォーマンス監視 =====
    
    // 長時間実行の検出
    if (window.performance && window.performance.mark) {
        window.addEventListener('load', function() {
            performance.mark('pageLoaded');
            
            // ナビゲーションタイミングの取得
            const perfData = performance.getEntriesByType('navigation')[0];
            if (perfData) {
                const loadTime = perfData.loadEventEnd - perfData.fetchStart;
                if (loadTime > 3000) {
                    console.warn(`Page load time: ${loadTime}ms - Consider optimization`);
                }
            }
        });
    }
    
})();