<?php
// SNS投稿作成ページ
?>

<div class="page-header">
    <h1><i class="bi bi-share"></i> SNS投稿作成</h1>
    <p class="text-muted">記事からSNS投稿を生成します</p>
</div>

<?php
// APIキーの設定状態をチェック
$missingKeys = [];
if (empty($config['api_keys']['openai'] ?? '') && empty($config['openai_api_key'] ?? '')) $missingKeys[] = 'OpenAI';
if (empty($config['api_keys']['claude'] ?? '') && empty($config['anthropic_api_key'] ?? '')) $missingKeys[] = 'Anthropic';
if (empty($config['api_keys']['gemini'] ?? '') && empty($config['google_api_key'] ?? '')) $missingKeys[] = 'Google (Gemini)';
if (empty($config['api_keys']['venice'] ?? '') && empty($config['venice_api_key'] ?? '')) $missingKeys[] = 'Venice AI';
if (empty($config['api_keys']['perplexity'] ?? '') && empty($config['perplexity_api_key'] ?? '')) $missingKeys[] = 'Perplexity';

if (!empty($missingKeys)):
?>
<div class="alert alert-warning alert-dismissible fade show" role="alert">
    <i class="bi bi-exclamation-triangle"></i> 
    <strong>APIキーが未設定です:</strong> <?php echo implode(', ', $missingKeys); ?>
    <br>
    <small>設定メニューからAPIキーを登録してください。未設定のAIモデルは使用できません。</small>
    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
</div>
<?php endif; ?>

<div class="row">
    <div class="col-lg-8">
        <div class="card">
            <div class="card-body">
                <form id="snsForm">
                    <!-- 本文入力 -->
                    <div class="mb-4">
                        <label for="articleContent" class="form-label">記事本文 <span class="text-danger">*</span></label>
                        <textarea class="form-control" id="articleContent" rows="8" 
                                  placeholder="SNS投稿にしたい記事の本文を貼り付けてください" required></textarea>
                        <small class="text-muted">文字数: <span id="charCount">0</span>文字</small>
                    </div>
                    
                    <!-- URL入力 -->
                    <div class="mb-4">
                        <label for="targetUrl" class="form-label">誘導先URL <span class="text-danger">*</span></label>
                        <input type="url" class="form-control" id="targetUrl" 
                               placeholder="https://example.com/article" required>
                    </div>
                    
                    <!-- 文字数設定 -->
                    <div class="row mb-4">
                        <div class="col-md-6">
                            <label for="maxLength" class="form-label">最大文字数 <span class="text-danger">*</span></label>
                            <input type="number" class="form-control" id="maxLength" 
                                   value="280" min="50" required>
                            <small class="text-muted">Twitter: 280文字、Facebook: 500文字推奨</small>
                        </div>
                        <div class="col-md-6">
                            <label for="usePerplexity" class="form-label">Yahoo!トレンド活用</label>
                            <div class="form-check mt-2">
                                <input class="form-check-input" type="checkbox" id="usePerplexity" checked>
                                <label class="form-check-label" for="usePerplexity">
                                    Perplexityを使用してYahoo!トレンドを取得
                                </label>
                            </div>
                        </div>
                    </div>
                    
                    <!-- モデル選択 -->
                    <div class="mb-4">
                        <label for="model" class="form-label">AIモデル</label>
                        <select class="form-control" id="model">
                            <?php foreach ($config['models'] as $provider => $models): ?>
                                <?php
                                // プロバイダーのAPIキーが設定されているかチェック
                                $apiKeyMap = [
                                    'openai' => ['api_keys' => 'openai', 'direct' => 'openai_api_key'],
                                    'anthropic' => ['api_keys' => 'claude', 'direct' => 'anthropic_api_key'],
                                    'google' => ['api_keys' => 'gemini', 'direct' => 'google_api_key'],
                                    'venice' => ['api_keys' => 'venice', 'direct' => 'venice_api_key'],
                                    'perplexity' => ['api_keys' => 'perplexity', 'direct' => 'perplexity_api_key']
                                ];
                                $hasApiKey = false;
                                if (isset($apiKeyMap[$provider])) {
                                    $apiKeysKey = $apiKeyMap[$provider]['api_keys'];
                                    $directKey = $apiKeyMap[$provider]['direct'];
                                    $hasApiKey = !empty($config['api_keys'][$apiKeysKey] ?? '') || !empty($config[$directKey] ?? '');
                                }
                                ?>
                                <optgroup label="<?php echo ucfirst($provider) . (!$hasApiKey ? ' (APIキー未設定)' : ''); ?>">
                                    <?php foreach ($models as $key => $model): ?>
                                        <?php 
                                        // THINKモデル（QwenとDeepSeek）を除外
                                        if (in_array($key, ['qwen3-235b', 'qwen-2.5-qwq-32b', 'deepseek-r1-671b'])) {
                                            continue;
                                        }
                                        ?>
                                        <option value="<?php echo htmlspecialchars($key); ?>" 
                                                <?php echo !$hasApiKey ? 'disabled' : ''; ?>
                                                <?php echo $key === $config['default_model'] && $hasApiKey ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($model['name']) . (!$hasApiKey ? ' (利用不可)' : ''); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </optgroup>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <!-- 生成ボタン -->
                    <div class="text-center">
                        <button type="submit" class="btn btn-primary btn-lg" id="generateBtn">
                            <i class="bi bi-lightning"></i> SNS投稿を生成
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- サイドバー -->
    <div class="col-lg-4">
        <div class="card mb-3">
            <div class="card-body">
                <h5 class="card-title"><i class="bi bi-info-circle"></i> 使い方</h5>
                <ol class="small">
                    <li>記事の本文を貼り付ける</li>
                    <li>誘導したいURLを入力</li>
                    <li>SNS投稿の最大文字数を設定</li>
                    <li>使用するAIモデルを選択</li>
                    <li>「SNS投稿を生成」ボタンをクリック</li>
                </ol>
                <p class="small text-muted mb-0">
                    5パターンの投稿が生成されます。各投稿は指定文字数ギリギリまで最適化されます。
                </p>
            </div>
        </div>
        
        <div class="card">
            <div class="card-body">
                <h5 class="card-title"><i class="bi bi-lightbulb"></i> ヒント</h5>
                <ul class="small mb-0">
                    <li><strong>Twitter:</strong> 280文字以内</li>
                    <li><strong>Facebook:</strong> 500文字推奨</li>
                    <li><strong>Instagram:</strong> 2,200文字まで可能</li>
                    <li class="mt-2">Perplexityを有効にすると、Yahoo!トレンドの最新キーワードを活用します</li>
                </ul>
            </div>
        </div>
    </div>
</div>

<!-- 結果表示エリア -->
<div id="resultsArea" class="mt-4" style="display: none;">
    <div class="card">
        <div class="card-header">
            <h5 class="mb-0"><i class="bi bi-chat-square-text"></i> 生成結果</h5>
        </div>
        <div class="card-body" id="resultsContent" style="max-width: 800px; margin: 0 auto;">
            <!-- 結果がここに表示される -->
        </div>
    </div>
</div>

<script>
// 文字数カウント
document.getElementById('articleContent').addEventListener('input', function() {
    const count = this.value.length;
    document.getElementById('charCount').textContent = count.toLocaleString();
});

// フォーム送信処理
document.getElementById('snsForm').addEventListener('submit', async function(e) {
    e.preventDefault();
    
    const btn = document.getElementById('generateBtn');
    const originalText = btn.innerHTML;
    btn.disabled = true;
    btn.innerHTML = '<span class="spinner-border spinner-border-sm"></span> 生成中...';
    
    try {
        const formData = {
            article_content: document.getElementById('articleContent').value,
            target_url: document.getElementById('targetUrl').value,
            max_length: parseInt(document.getElementById('maxLength').value),
            model: document.getElementById('model').value,
            use_perplexity: document.getElementById('usePerplexity').checked
        };
        
        // 記事本文の文字数を確認
        console.log('記事本文の文字数: ' + formData.article_content.length + '文字');
        
        // Perplexityでトレンドを取得（有効な場合）
        let trendKeywords = '';
        if (formData.use_perplexity) {
            try {
                const trendResponse = await fetch('../api/service.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        action: 'generate',
                        title: 'トレンド取得',
                        keywords: '',
                        model: 'sonar-reasoning-pro',
                        system_prompt: 'あなたはトレンド情報を提供するアシスタントです。',
                        user_prompt: '現在のYahoo!トレンドキーワード上位20個を取得',
                        temperature: 0.1,
                        max_tokens: 500,
                        output_format: 'text',
                        enable_images: false,
                        enable_perplexity: false
                    })
                });
                
                if (trendResponse.ok) {
                    const trendResult = await trendResponse.json();
                    if (trendResult.content) {
                        trendKeywords = '\n\nYahoo!トレンド情報:\n' + trendResult.content;
                    }
                }
            } catch (e) {
                console.log('トレンド取得エラー:', e);
            }
        }
        
        // 記事生成と全く同じ形式でservice.phpを呼び出す
        const response = await fetch('../api/service.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                action: 'generate_sns_advanced',
                article_content: formData.article_content,
                target_url: formData.target_url,
                max_length: formData.max_length,
                model: formData.model,
                use_perplexity: formData.use_perplexity
            })
        });
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('API Error:', errorText);
            
            // JSONエラーレスポンスを解析
            let errorMessage = `通信エラーが発生しました (${response.status})`;
            try {
                const errorJson = JSON.parse(errorText);
                if (errorJson.error) {
                    errorMessage = errorJson.error;
                    
                    // Gemini特有のエラーメッセージ
                    if (formData.model.startsWith('gemini') && errorMessage.includes('APIキーが設定されていません')) {
                        errorMessage = 'Gemini APIキーが設定されていません。設定メニューからGoogle APIキーを登録してください。';
                    }
                    // MAX_TOKENSエラーの場合
                    if (errorMessage.includes('MAX_TOKENS')) {
                        errorMessage = 'トークン数が上限に達しました。記事の内容を短くするか、別のモデルを使用してください。';
                    }
                    // Anthropic過負荷エラーの場合
                    if (errorMessage.includes('Overloaded')) {
                        errorMessage = 'Anthropic APIが過負荷状態です。しばらく待ってから再試行するか、別のモデル（GPT-4など）を使用してください。';
                    }
                }
            } catch (e) {
                // JSONパースエラーの場合はそのままエラーテキストを使用
                errorMessage += `: ${errorText}`;
            }
            
            throw new Error(errorMessage);
        }
        
        const result = await response.json();
        
        console.log('API Response:', result); // デバッグ用
        console.log('Selected Model:', formData.model); // モデル名を確認
        
        if (result.posts) {
            // SNS投稿APIのレスポンス形式に対応
            console.log('Posts:', result.posts); // デバッグ用
            displayResults({ posts: result.posts, model: formData.model });
        } else if (result.content) {
            // 記事生成のレスポンス形式（後方互換性）
            const content = result.content;
            console.log('Content:', content); // デバッグ用
            
            const posts = [];
            const lines = content.split('\n');
            let currentPost = '';
            let skipSummary = false;
            
            for (const line of lines) {
                const trimmed = line.trim();
                
                // 要約行の検出
                if (trimmed.startsWith('要約') || trimmed.startsWith('要約：')) {
                    skipSummary = true;
                    continue;
                }
                
                // 要約後の空行をスキップ
                if (skipSummary && trimmed === '') {
                    skipSummary = false;
                    continue;
                }
                
                // 「–」で始まる行は新しい投稿
                if (trimmed.startsWith('–') || trimmed.startsWith('-')) {
                    if (currentPost) {
                        posts.push(currentPost.trim());
                    }
                    currentPost = trimmed.substring(1).trim();
                } else if (currentPost && trimmed) {
                    currentPost += '\n' + trimmed;
                }
            }
            
            // 最後の投稿を追加
            if (currentPost) {
                posts.push(currentPost.trim());
            }
            
            console.log('Parsed posts:', posts); // デバッグ用
            console.log('Number of posts found:', posts.length);
            
            // postsが空の場合、エラーではなく空の結果として扱う
            displayResults({ posts: posts, model: formData.model });
        } else {
            alert('エラー: ' + (result.error || 'SNS投稿の生成に失敗しました'));
        }
        
    } catch (error) {
        alert('エラー: ' + error.message);
    } finally {
        btn.disabled = false;
        btn.innerHTML = originalText;
    }
});

// 結果表示
function displayResults(result) {
    const resultsArea = document.getElementById('resultsArea');
    const resultsContent = document.getElementById('resultsContent');
    
    let html = '';
    
    if (result.posts && result.posts.length > 0) {
        // スマートなカード表示
        const colors = ['primary', 'success', 'info', 'warning', 'secondary'];
        const ctaTypes = {
            '続きを読む👉': 'primary',
            '詳細はこちら': 'info',
            '今すぐ確認': 'success'
        };
        
        result.posts.forEach((post, index) => {
            const charCount = post.length;
            const maxLength = parseInt(document.getElementById('maxLength').value);
            const percentage = Math.round((charCount / maxLength) * 100);
            const color = colors[index % colors.length];
            
            // CTA判定
            let ctaBadge = '';
            for (const [cta, badgeColor] of Object.entries(ctaTypes)) {
                if (post.includes(cta)) {
                    ctaBadge = `<span class="badge bg-${badgeColor} ms-2">${cta}</span>`;
                    break;
                }
            }
            
            html += `
                <div style="margin-bottom: 1rem;">
                    <div style="border: 1px solid #dee2e6; border-radius: 0.375rem; background-color: white; overflow: hidden;">
                        <div style="background-color: #f8f9fa; border-bottom: 1px solid #dee2e6; padding: 0.5rem 0.75rem;">
                            <div style="display: flex; justify-content: space-between; align-items: center;">
                                <div>
                                    <h6 style="margin: 0; font-size: 0.9rem; font-weight: 600;">
                                        <span style="color: ${color === 'primary' ? '#0d6efd' : color === 'success' ? '#198754' : color === 'info' ? '#0dcaf0' : color === 'warning' ? '#ffc107' : '#6c757d'};">●</span> 
                                        パターン ${index + 1}
                                        ${ctaBadge}
                                    </h6>
                                </div>
                                <div>
                                    <span style="display: inline-block; padding: 0.2rem 0.4rem; font-size: 0.75rem; border-radius: 0.25rem; background-color: ${percentage >= 95 ? '#198754' : percentage >= 80 ? '#ffc107' : '#dc3545'}; color: white;">
                                        ${charCount}/${maxLength}文字
                                    </span>
                                    <span style="display: inline-block; padding: 0.2rem 0.4rem; font-size: 0.75rem; border-radius: 0.25rem; background-color: #6c757d; color: white; margin-left: 0.25rem;">
                                        ${percentage}%
                                    </span>
                                </div>
                            </div>
                        </div>
                        <div style="padding: 0.75rem;">
                            <div style="margin-bottom: 0.75rem; font-size: 0.9rem; line-height: 1.5; color: #212529; white-space: pre-wrap; word-wrap: break-word;">
${escapeHtml(post)}
                            </div>
                            <div style="display: flex; gap: 0.5rem;">
                                <button style="padding: 0.25rem 0.5rem; font-size: 0.8rem; background-color: ${color === 'primary' ? '#0d6efd' : color === 'success' ? '#198754' : color === 'info' ? '#0dcaf0' : color === 'warning' ? '#ffc107' : '#6c757d'}; color: white; border: none; border-radius: 0.25rem; cursor: pointer;" onclick="copyPost('${btoa(encodeURIComponent(post))}')">
                                    コピー
                                </button>
                                <button style="padding: 0.25rem 0.5rem; font-size: 0.8rem; background-color: transparent; color: ${color === 'primary' ? '#0d6efd' : color === 'success' ? '#198754' : color === 'info' ? '#0dcaf0' : color === 'warning' ? '#ffc107' : '#6c757d'}; border: 1px solid ${color === 'primary' ? '#0d6efd' : color === 'success' ? '#198754' : color === 'info' ? '#0dcaf0' : color === 'warning' ? '#ffc107' : '#6c757d'}; border-radius: 0.25rem; cursor: pointer;" onclick="previewPost('${btoa(encodeURIComponent(post))}')">
                                    プレビュー
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            `;
        });
        
        // グリッドレイアウトは削除（シンプルに表示）
        
        if (result.trends && result.trends.length > 0) {
            html = `
                <div class="alert alert-info mb-3">
                    <strong>使用したYahoo!トレンドキーワード:</strong> ${result.trends.join(', ')}
                </div>
            ` + html;
        }
    } else {
        html = '<p class="text-center text-muted">生成結果がありません</p>';
    }
    
    resultsContent.innerHTML = html;
    resultsArea.style.display = 'block';
    
    // 結果エリアまでスクロール
    resultsArea.scrollIntoView({ behavior: 'smooth', block: 'start' });
}

// HTMLエスケープ
function escapeHtml(text) {
    const map = {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#039;'
    };
    return text.replace(/[&<>"']/g, m => map[m]);
}

// コピー機能
async function copyPost(encodedPost) {
    try {
        const post = decodeURIComponent(atob(encodedPost));
        await navigator.clipboard.writeText(post);
        showToast('SNS投稿をコピーしました', 'success');
    } catch (error) {
        // フォールバック
        const textarea = document.createElement('textarea');
        textarea.value = decodeURIComponent(atob(encodedPost));
        textarea.style.position = 'fixed';
        textarea.style.opacity = '0';
        document.body.appendChild(textarea);
        textarea.select();
        document.execCommand('copy');
        document.body.removeChild(textarea);
        showToast('SNS投稿をコピーしました', 'success');
    }
}

// プレビュー機能
function previewPost(encodedPost) {
    const post = decodeURIComponent(atob(encodedPost));
    
    // 既存のモーダルを削除
    const existingModal = document.getElementById('snsPreviewModal');
    if (existingModal) {
        existingModal.remove();
    }
    
    // モーダル背景
    const modalBackdrop = document.createElement('div');
    modalBackdrop.className = 'modal-backdrop fade show';
    modalBackdrop.style.zIndex = '1040';
    
    // モーダル本体
    const modal = document.createElement('div');
    modal.id = 'snsPreviewModal';
    modal.className = 'modal fade show';
    modal.style.display = 'block';
    modal.style.zIndex = '1050';
    modal.innerHTML = `
        <div class="modal-dialog" style="max-width: 500px;">
            <div class="modal-content" style="overflow: hidden;">
                <div class="modal-header py-2 px-3" style="border-bottom: 1px solid #dee2e6;">
                    <h6 class="modal-title mb-0">SNS投稿プレビュー</h6>
                    <button type="button" class="btn-close btn-sm" onclick="closeSnsModal()"></button>
                </div>
                <div class="modal-body p-2">
                    <div class="card border-0 shadow-sm mb-0">
                        <div class="card-body p-2" style="font-family: system-ui, -apple-system;">
                            <div class="d-flex mb-2">
                                <div class="bg-primary rounded-circle" style="width: 36px; height: 36px;"></div>
                                <div class="ms-2">
                                    <div class="fw-bold small">あなたの名前</div>
                                    <div class="text-muted" style="font-size: 0.75rem;">@username</div>
                                </div>
                            </div>
                            <div style="white-space: pre-wrap; line-height: 1.4; font-size: 0.95rem;">${escapeHtml(post)}</div>
                            <div class="text-muted mt-2" style="font-size: 0.75rem;">
                                <i class="bi bi-heart"></i> 0
                                <i class="bi bi-chat ms-2"></i> 0
                                <i class="bi bi-arrow-repeat ms-2"></i> 0
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer py-2 px-3" style="border-top: 1px solid #dee2e6;">
                    <button type="button" class="btn btn-sm btn-secondary" onclick="closeSnsModal()">閉じる</button>
                    <button type="button" class="btn btn-sm btn-primary" onclick="copyPost('${btoa(encodeURIComponent(post))}'); closeSnsModal();">
                        <i class="bi bi-clipboard"></i> コピー
                    </button>
                </div>
            </div>
        </div>
    `;
    
    document.body.appendChild(modalBackdrop);
    document.body.appendChild(modal);
}

// モーダルを閉じる関数
function closeSnsModal() {
    const modal = document.getElementById('snsPreviewModal');
    const backdrop = document.querySelector('.modal-backdrop');
    if (modal) modal.remove();
    if (backdrop) backdrop.remove();
}

// トースト表示
function showToast(message, type) {
    const toastHtml = `
        <div class="position-fixed top-0 end-0 p-3" style="z-index: 11">
            <div class="toast show align-items-center text-white bg-${type}" role="alert">
                <div class="d-flex">
                    <div class="toast-body">${message}</div>
                    <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
                </div>
            </div>
        </div>
    `;
    
    const toastContainer = document.createElement('div');
    toastContainer.innerHTML = toastHtml;
    document.body.appendChild(toastContainer);
    
    setTimeout(() => {
        toastContainer.remove();
    }, 3000);
}
</script>