<?php
// セッションチェックは親ファイル（index.php）で行われているため、ここでは不要

// 保存済みのサイト設定を読み込み
$siteConfigFile = __DIR__ . '/../../data/site-configs.json';
$siteConfigs = [];
if (file_exists($siteConfigFile)) {
    $siteConfigs = json_decode(file_get_contents($siteConfigFile), true) ?: [];
}

// FTPプロファイルを読み込み
$configFile = __DIR__ . '/../../config.php';
$ftpProfiles = [];
if (file_exists($configFile)) {
    $configData = require $configFile;
    $ftpProfiles = $configData['ftp_profiles'] ?? [];
}

// 記事データを読み込み
$historyFile = __DIR__ . '/../../data/history.json';
$articles = file_exists($historyFile) ? json_decode(file_get_contents($historyFile), true) : [];
?>

<style>
/* FTPモーダル内部のレイアウト最適化 - <?php echo date('Y-m-d H:i:s'); ?> */
#ftpSelectModal .modal-dialog {
    /* ダイアログのサイズを固定 */
    width: 450px !important;
    max-width: 90vw !important;
}

#ftpSelectModal .modal-content {
    /* コンテンツに応じて高さを自動調整 */
    height: auto !important;
    max-height: none !important;
}

#ftpSelectModal .modal-header {
    padding: 1rem 1.25rem;
    border-bottom: 1px solid #dee2e6;
}

#ftpSelectModal .modal-body {
    padding: 1.25rem !important;
    /* 最小高さを設定しない */
    min-height: auto !important;
    /* 固定高さを完全に削除 */
    max-height: none !important;
    height: auto !important;
    overflow: visible !important;
    /* 背景色を追加してデバッグ */
    background-color: #ffffff;
}

/* FTPプロファイルが少ない場合は高さを自動調整 */
#ftpSelectModal .list-group {
    margin-bottom: 0;
    /* スクロールを削除 */
    max-height: none !important;
    overflow: visible !important;
}

/* FTPプロファイルが5個以上の場合のみスクロール */
#ftpSelectModal .list-group:has(.ftp-profile-item:nth-child(5)) {
    max-height: 400px !important;
    overflow-y: auto !important;
}

.ftp-profile-item {
    padding: 0.75rem 1rem;
    border: 1px solid #dee2e6;
    border-radius: 0.375rem;
    margin-bottom: 0.5rem;
    transition: all 0.2s ease;
}

.ftp-profile-item:last-child {
    margin-bottom: 0;
}

.ftp-profile-item:hover {
    background-color: #e9ecef;
    border-color: #0d6efd;
}

.ftp-profile-item.active {
    background-color: #0d6efd;
    border-color: #0d6efd;
    color: white;
}

.ftp-profile-item.active .text-muted {
    color: rgba(255, 255, 255, 0.8) !important;
}

.ftp-profile-item.active .bi {
    color: white;
}

.ftp-profile-item h6 {
    font-size: 0.9375rem;
    font-weight: 600;
    margin-bottom: 0.125rem;
}

.ftp-profile-item small {
    font-size: 0.8125rem;
}

.ftp-profile-item .bi {
    font-size: 1.125rem;
    color: #0d6efd;
}

/* モーダルフッター */
#ftpSelectModal .modal-footer {
    padding: 0.75rem 1.25rem;
    background-color: #f8f9fa;
    flex-shrink: 0;
}

/* FTP設定がない場合のスタイル */
#ftpSelectModal .text-center {
    padding: 1.5rem 0;
}

#ftpSelectModal .text-center p {
    margin-bottom: 1rem;
    font-size: 0.9375rem;
    color: #6c757d;
}

/* デザイン設定モーダルのタブコンテンツ高さ制限 */
#designSettingsModal .tab-content {
    max-height: 60vh;
    overflow-y: auto;
}

#designSettingsModal .tab-pane {
    padding-top: 1rem;
}

/* ヘッダータイプ選択後の設定エリア */
#designSettingsModal .header-type-settings {
    margin-top: 1rem !important;
}
</style>

<div class="container mt-4">
    <h2>サイト更新（ワンクリック）</h2>
    
    <div class="alert alert-info">
        <i class="bi bi-info-circle"></i> 保存済みの設定を使用して、ワンクリックでサイトを更新・デプロイできます。
    </div>
    
    <!-- 記事管理パネル -->
    <div class="card mb-4">
        <div class="card-header bg-warning text-dark">
            <h5 class="mb-0"><i class="bi bi-folder-plus"></i> 記事の管理</h5>
        </div>
        <div class="card-body">
            <p>現在、システム全体で <strong><?php echo count($articles); ?></strong> 件の記事があります。</p>
            <p class="text-muted mb-3">各サイトの「記事を管理」ボタンから、どの記事をどのサイトに含めるかを設定できます。</p>
        </div>
    </div>
    
    <?php if (empty($siteConfigs)): ?>
    <div class="alert alert-warning">
        <h4>サイト設定がありません</h4>
        <p>まず、サイト生成ページで初回設定を行ってください。</p>
        <a href="?page=site-generator" class="btn btn-primary">サイト生成ページへ</a>
    </div>
    <?php else: ?>
    
    <!-- 保存済みサイト設定一覧 -->
    <div class="row">
        <?php foreach ($siteConfigs as $configId => $siteConfig): ?>
        <div class="col-md-6 mb-4">
            <div class="card h-100">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0"><?php echo htmlspecialchars($siteConfig['display_name']); ?></h5>
                </div>
                <div class="card-body">
                    <p class="mb-2"><strong>URL:</strong> 
                        <?php 
                        $siteUrl = $siteConfig['url'];
                        // サブディレクトリがある場合はURLに追加
                        if ($siteConfig['deployment_type'] === 'subdirectory' && !empty($siteConfig['subdirectory'])) {
                            $siteUrl = rtrim($siteUrl, '/') . '/' . ltrim($siteConfig['subdirectory'], '/');
                        }
                        ?>
                        <a href="<?php echo htmlspecialchars($siteUrl); ?>" target="_blank" rel="noopener noreferrer" class="text-decoration-none">
                            <?php echo htmlspecialchars($siteUrl); ?> 
                            <i class="bi bi-box-arrow-up-right small"></i>
                        </a>
                    </p>
                    <p class="mb-2"><strong>配置:</strong> <?php echo $siteConfig['deployment_type'] === 'root' ? 'ルート' : 'サブディレクトリ（' . $siteConfig['subdirectory'] . '）'; ?></p>
                    <p class="mb-2"><strong>記事数:</strong> 
                        <span class="badge bg-info">
                            <?php 
                            $articleCount = isset($siteConfig['article_ids']) ? count($siteConfig['article_ids']) : 0;
                            echo $articleCount; 
                            ?>記事
                        </span>
                    </p>
                    <p class="mb-2"><strong>FTP設定:</strong> 
                        <?php if (isset($siteConfig['ftp_profile_id']) && isset($ftpProfiles[$siteConfig['ftp_profile_id']])): ?>
                            <span class="badge bg-success"><?php echo htmlspecialchars($ftpProfiles[$siteConfig['ftp_profile_id']]['name']); ?></span>
                            <button class="btn btn-sm btn-outline-secondary ms-2 select-ftp-btn" 
                                    data-config-id="<?php echo htmlspecialchars($configId); ?>"
                                    data-site-name="<?php echo htmlspecialchars($siteConfig['display_name']); ?>">
                                <i class="bi bi-pencil"></i>
                            </button>
                        <?php else: ?>
                            <span class="badge bg-secondary">未設定</span>
                            <?php if (!empty($ftpProfiles)): ?>
                            <button class="btn btn-sm btn-outline-primary ms-2 select-ftp-btn" 
                                    data-config-id="<?php echo htmlspecialchars($configId); ?>"
                                    data-site-name="<?php echo htmlspecialchars($siteConfig['display_name']); ?>">
                                <i class="bi bi-plus-circle"></i> 選択
                            </button>
                            <?php endif; ?>
                        <?php endif; ?>
                    </p>
                    <p class="mb-3"><small class="text-muted">最終更新: <?php echo $siteConfig['last_updated'] ?? '未更新'; ?></small></p>
                    
                    <div class="d-grid gap-2">
                        <button class="btn btn-warning manage-articles-btn" 
                                data-config-id="<?php echo htmlspecialchars($configId); ?>"
                                data-site-name="<?php echo htmlspecialchars($siteConfig['display_name']); ?>">
                            <i class="bi bi-folder-plus"></i> 記事を管理
                        </button>
                        <button class="btn btn-primary btn-lg update-site-btn" 
                                data-config-id="<?php echo htmlspecialchars($configId); ?>"
                                data-site-name="<?php echo htmlspecialchars($siteConfig['display_name']); ?>">
                            <i class="bi bi-arrow-clockwise"></i> サイトを更新
                        </button>
                        <?php if (isset($siteConfig['ftp_profile_id']) && isset($ftpProfiles[$siteConfig['ftp_profile_id']])): ?>
                        <button class="btn btn-success update-and-deploy-btn" 
                                data-config-id="<?php echo htmlspecialchars($configId); ?>"
                                data-site-name="<?php echo htmlspecialchars($siteConfig['display_name']); ?>">
                            <i class="bi bi-cloud-upload"></i> 更新してデプロイ
                        </button>
                        <?php endif; ?>
                        <div class="btn-group" role="group">
                            <button class="btn btn-outline-info btn-sm design-settings-btn"
                                    data-config-id="<?php echo htmlspecialchars($configId); ?>"
                                    data-site-name="<?php echo htmlspecialchars($siteConfig['display_name']); ?>">
                                <i class="bi bi-palette"></i> デザイン設定
                            </button>
                            <button class="btn btn-outline-secondary btn-sm edit-config-btn"
                                    data-config-id="<?php echo htmlspecialchars($configId); ?>">
                                <i class="bi bi-gear"></i> 基本設定
                            </button>
                            <button class="btn btn-outline-danger btn-sm delete-config-btn"
                                    data-config-id="<?php echo htmlspecialchars($configId); ?>"
                                    data-site-name="<?php echo htmlspecialchars($siteConfig['display_name']); ?>">
                                <i class="bi bi-trash"></i> 削除
                            </button>
                        </div>
                    </div>
                </div>
                <div class="card-footer">
                    <div class="update-status" id="status-<?php echo htmlspecialchars($configId); ?>"></div>
                </div>
            </div>
        </div>
        <?php endforeach; ?>
    </div>
    
    <!-- 新しい記事の通知 -->
    <div class="card mt-4">
        <div class="card-body">
            <h5><i class="bi bi-bell"></i> 最新の記事</h5>
            <?php
            $latestArticles = array_slice($articles, 0, 5);
            if (!empty($latestArticles)):
            ?>
            <p>最近作成された記事：</p>
            <ul class="list-unstyled">
                <?php foreach ($latestArticles as $article): ?>
                <li class="mb-1">
                    <i class="bi bi-file-text"></i> 
                    <?php echo htmlspecialchars($article['title']); ?>
                    <small class="text-muted">(<?php echo date('Y/m/d H:i', strtotime($article['created_at'])); ?>)</small>
                </li>
                <?php endforeach; ?>
            </ul>
            <p class="text-muted"><small>※ 各サイトに含める記事は「記事を管理」から設定してください。</small></p>
            <?php else: ?>
            <p class="text-muted">記事がありません。</p>
            <?php endif; ?>
        </div>
    </div>
    
    <?php endif; ?>
</div>

<!-- 記事管理モーダル -->
<div class="modal fade" id="articleManageModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="articleManageTitle">記事の管理</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="alert alert-info">
                    <i class="bi bi-info-circle"></i> このサイトに含める記事を選択してください。
                </div>
                
                <!-- カテゴリーフィルター -->
                <div class="row mb-3">
                    <div class="col-md-6">
                        <label class="form-label">カテゴリーで絞り込み</label>
                        <select class="form-control" id="modalCategoryFilter">
                            <option value="">すべて</option>
                            <?php 
                            $categories = array_unique(array_column($articles, 'category'));
                            foreach ($categories as $category): 
                            ?>
                            <option value="<?php echo htmlspecialchars($category); ?>">
                                <?php echo htmlspecialchars($category); ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">選択状態</label>
                        <p class="form-control-plaintext">
                            <span id="modalSelectedCount">0</span> / <?php echo count($articles); ?> 記事を選択中
                        </p>
                    </div>
                </div>
                
                <!-- 記事リスト -->
                <div class="article-selection-modal" style="max-height: 400px; overflow-y: auto; border: 1px solid #ddd; padding: 10px;">
                    <div class="form-check mb-2">
                        <input type="checkbox" class="form-check-input" id="modalSelectAll">
                        <label class="form-check-label" for="modalSelectAll">
                            <strong>すべて選択</strong>
                        </label>
                    </div>
                    <hr>
                    <div id="modalArticleList">
                        <!-- 動的に生成 -->
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">キャンセル</button>
                <button type="button" class="btn btn-primary" id="saveArticleSelection">
                    <i class="bi bi-save"></i> 選択を保存
                </button>
            </div>
        </div>
    </div>
</div>

<!-- FTPプロファイル選択モーダル -->
<div class="modal fade" id="ftpSelectModal" tabindex="-1">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="ftpSelectTitle">FTP設定を選択</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <?php if (!empty($ftpProfiles)): ?>
                <div class="list-group" id="ftpProfileList">
                    <?php foreach ($ftpProfiles as $profileId => $profile): ?>
                    <button type="button" class="list-group-item list-group-item-action ftp-profile-item" 
                            data-profile-id="<?php echo htmlspecialchars($profileId); ?>">
                        <div class="d-flex justify-content-between align-items-center" style="max-height: 60px;">
                            <div class="text-truncate me-2">
                                <h6 class="mb-0"><?php echo htmlspecialchars($profile['name']); ?></h6>
                                <small class="text-muted text-truncate d-block"><?php echo htmlspecialchars($profile['host']); ?> - <?php echo htmlspecialchars($profile['username']); ?></small>
                            </div>
                            <i class="bi bi-hdd-network flex-shrink-0"></i>
                        </div>
                    </button>
                    <?php endforeach; ?>
                </div>
                <?php else: ?>
                <div class="text-center">
                    <p class="text-muted">FTP設定がありません。</p>
                    <a href="?page=ftp-settings" class="btn btn-primary">
                        <i class="bi bi-plus-circle"></i> FTP設定を追加
                    </a>
                </div>
                <?php endif; ?>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">キャンセル</button>
            </div>
        </div>
    </div>
</div>

<!-- デザイン設定モーダル -->
<div class="modal fade" id="designSettingsModal" tabindex="-1">
    <div class="modal-dialog modal-xl modal-dialog-scrollable">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="designSettingsTitle">デザイン設定</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" style="max-height: 70vh; overflow-y: auto;">
                <form id="designSettingsForm">
                    <!-- タブナビゲーション -->
                    <ul class="nav nav-tabs mb-3" role="tablist">
                        <li class="nav-item" role="presentation">
                            <button class="nav-link active" data-bs-toggle="tab" data-bs-target="#basicInfoTab" type="button">
                                <i class="bi bi-info-circle"></i> 基本情報
                            </button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" data-bs-toggle="tab" data-bs-target="#freeSpaceTab" type="button">
                                <i class="bi bi-textarea"></i> フリースペース
                            </button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" data-bs-toggle="tab" data-bs-target="#layoutTab" type="button">
                                <i class="bi bi-grid"></i> レイアウト
                            </button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" data-bs-toggle="tab" data-bs-target="#headerDesignTab" type="button">
                                <i class="bi bi-layout-text-window-reverse"></i> ヘッダーデザイン
                            </button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" data-bs-toggle="tab" data-bs-target="#headingTab" type="button">
                                <i class="bi bi-type-h1"></i> 見出しデザイン
                            </button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" data-bs-toggle="tab" data-bs-target="#listStyleTab" type="button">
                                <i class="bi bi-list-ul"></i> リストデザイン
                            </button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" data-bs-toggle="tab" data-bs-target="#backgroundTab" type="button">
                                <i class="bi bi-palette"></i> 背景設定
                            </button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" data-bs-toggle="tab" data-bs-target="#themeColorTab" type="button">
                                <i class="bi bi-paint-bucket"></i> テーマカラー
                            </button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" data-bs-toggle="tab" data-bs-target="#verificationTab" type="button">
                                <i class="bi bi-shield-check"></i> 検証・トラッキング
                            </button>
                        </li>
                    </ul>
                    
                    <!-- タブコンテンツ -->
                    <div class="tab-content" style="max-height: 60vh; overflow-y: auto;">
                        <!-- 基本情報タブ -->
                        <div class="tab-pane fade show active" id="basicInfoTab" role="tabpanel">
                            <div class="row">
                                <div class="col-12">
                                    <div class="alert alert-info">
                                        <i class="bi bi-info-circle"></i> サイトの基本情報とSEO設定を管理します。
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">サイト名（表示名）</label>
                                        <input type="text" class="form-control" id="siteDisplayName" placeholder="例：私のおすすめブログ">
                                        <small class="text-muted">サイトのヘッダーやタイトルタグに表示される名前です</small>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">サイト説明文</label>
                                        <textarea class="form-control" id="siteDescription" rows="3" placeholder="例：最新のガジェットレビューと使い方をわかりやすく解説"></textarea>
                                        <small class="text-muted">サイトの説明文。メタディスクリプションにも使用されます</small>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">ヒーローキャッチフレーズ</label>
                                        <input type="text" class="form-control" id="heroCatchphrase" placeholder="例：あなたの毎日に、新しい発見を">
                                        <small class="text-muted">ヘッダーアニメーションの後ろに表示される大きな文字</small>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">SEOキーワード（メタキーワード）</label>
                                        <input type="text" class="form-control" id="siteKeywords" placeholder="例：ガジェット,レビュー,スマートフォン,タブレット">
                                        <small class="text-muted">サイト全体のSEOキーワードをカンマ区切りで入力（最大10個程度推奨）</small>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">メタディスクリプション（トップページ用）</label>
                                        <textarea class="form-control" id="siteMetaDescription" rows="3" placeholder="キャッチフレーズと異なる場合に入力"></textarea>
                                        <small class="text-muted">
                                            検索結果に表示される説明文（120文字程度推奨）
                                            <span id="metaDescLength" class="text-primary">0</span>文字
                                        </small>
                                    </div>
                                    
                                    <hr>
                                    
                                    <h6 class="mb-3"><i class="bi bi-globe"></i> OGP設定（SNSシェア時の表示）</h6>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">OGPタイトル</label>
                                        <input type="text" class="form-control" id="ogpTitle" placeholder="サイト名と同じ場合は空欄">
                                        <small class="text-muted">SNSでシェアされた時のタイトル</small>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="form-label">OGP説明文</label>
                                        <textarea class="form-control" id="ogpDescription" rows="2" placeholder="キャッチフレーズと同じ場合は空欄"></textarea>
                                        <small class="text-muted">SNSでシェアされた時の説明文</small>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- フリースペースタブ -->
                        <div class="tab-pane fade" id="freeSpaceTab" role="tabpanel">
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">トップページ上部フリースペース</label>
                                    <textarea class="form-control free-space" id="topUpperFreeSpace" rows="8" style="word-wrap: break-word; word-break: break-all; overflow-wrap: anywhere; max-width: 100%; overflow-x: auto; white-space: pre-wrap;" placeholder="HTMLコードを入力（広告、お知らせなど）"></textarea>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">トップページ下部フリースペース</label>
                                    <textarea class="form-control free-space" id="topLowerFreeSpace" rows="8" style="word-wrap: break-word; word-break: break-all; overflow-wrap: anywhere; max-width: 100%; overflow-x: auto; white-space: pre-wrap;" placeholder="HTMLコードを入力（広告、関連リンクなど）"></textarea>
                                </div>
                            </div>
                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">記事ページ上部フリースペース</label>
                                    <textarea class="form-control free-space" id="articleUpperFreeSpace" rows="8" style="word-wrap: break-word; word-break: break-all; overflow-wrap: anywhere; max-width: 100%; overflow-x: auto; white-space: pre-wrap;" placeholder="HTMLコードを入力（広告、お知らせなど）"></textarea>
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">記事ページ下部フリースペース</label>
                                    <textarea class="form-control free-space" id="articleLowerFreeSpace" rows="8" style="word-wrap: break-word; word-break: break-all; overflow-wrap: anywhere; max-width: 100%; overflow-x: auto; white-space: pre-wrap;" placeholder="HTMLコードを入力（広告、関連記事など）"></textarea>
                                </div>
                            </div>
                        </div>
                        
                        <!-- レイアウトタブ -->
                        <div class="tab-pane fade" id="layoutTab" role="tabpanel">
                            <div class="row">
                                <div class="col-12 mb-4">
                                    <h6><i class="bi bi-grid"></i> カラム数設定</h6>
                                    <div class="mb-3">
                                        <label class="form-label">記事一覧のカラム数</label>
                                        <select class="form-select" id="columnLayout">
                                            <option value="modern-blog">2カラム（レスポンシブ）</option>
                                            <option value="modern-blog-1col">1カラム（幅広）</option>
                                        </select>
                                        <small class="text-muted">
                                            <i class="bi bi-info-circle"></i> 
                                            2カラム：画面幅に応じて1〜2列に自動調整<br>
                                            1カラム：常に1列で表示（記事が大きく表示されます）
                                        </small>
                                    </div>
                                </div>
                                
                                <div class="col-12 mb-4">
                                    <h6><i class="bi bi-collection"></i> 表示件数設定</h6>
                                    <div class="mb-3">
                                        <label class="form-label">1ページあたりの記事数</label>
                                        <input type="number" class="form-control" id="articlesPerPage" min="1" max="50" value="10" placeholder="10">
                                        <small class="text-muted">
                                            <i class="bi bi-info-circle"></i> 
                                            トップページや一覧ページに表示する記事数（デフォルト: 10）
                                        </small>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- ヘッダーデザインタブ -->
                        <div class="tab-pane fade" id="headerDesignTab" role="tabpanel">
                            <div class="row">
                                <div class="col-12 mb-4">
                                    <h6><i class="bi bi-image"></i> ヘッダータイプ</h6>
                                    <div class="mb-3">
                                        <label class="form-label">ヘッダーデザインを選択</label>
                                        <select class="form-select" id="headerType">
                                            <option value="gradient">グラデーション（デフォルト）</option>
                                            <option value="image">画像</option>
                                            <option value="pattern-particles">アニメーション - パーティクル</option>
                                            <option value="pattern-waves">アニメーション - ウェーブ</option>
                                            <option value="pattern-geometric">アニメーション - ジオメトリック</option>
                                            <option value="pattern-stars">アニメーション - スターフィールド</option>
                                            <option value="pattern-aurora">アニメーション - オーロラ</option>
                                            <option value="pattern-matrix">アニメーション - マトリックス</option>
                                            <option value="pattern-bubbles">アニメーション - バブル</option>
                                            <option value="pattern-hexagon">アニメーション - ヘキサゴン</option>
                                            <option value="pattern-circuit">アニメーション - サーキット</option>
                                            <option value="pattern-galaxy">アニメーション - ギャラクシー</option>
                                            <option value="pattern-rain">アニメーション - レイン</option>
                                            <option value="pattern-fire">アニメーション - ファイア</option>
                                            <option value="pattern-snow">アニメーション - スノー</option>
                                            <option value="pattern-neural">アニメーション - ニューラル</option>
                                            <option value="pattern-mosaic">アニメーション - モザイク</option>
                                            <option value="pattern-sparkle">アニメーション - キラキラ</option>
                                            <option value="pattern-crystal">アニメーション - クリスタル</option>
                                            <option value="pattern-sakura">アニメーション - 桜吹雪</option>
                                            <option value="pattern-firefly">アニメーション - 蛍火</option>
                                            <option value="pattern-rainbow">アニメーション - レインボー</option>
                                            <option value="pattern-diamond">アニメーション - ダイヤモンド</option>
                                            <option value="pattern-glitter">アニメーション - グリッター</option>
                                            <option value="pattern-stardust">アニメーション - スターダスト</option>
                                        </select>
                                    </div>
                                    
                                    <!-- グラデーション設定 -->
                                    <div id="gradientSettings" class="header-type-settings">
                                        <h6 class="mt-4">グラデーション設定</h6>
                                        <div class="row">
                                            <div class="col-md-6 mb-3">
                                                <label class="form-label">開始色</label>
                                                <input type="color" class="form-control form-control-color" id="gradientStart" value="#667eea">
                                            </div>
                                            <div class="col-md-6 mb-3">
                                                <label class="form-label">終了色</label>
                                                <input type="color" class="form-control form-control-color" id="gradientEnd" value="#764ba2">
                                            </div>
                                        </div>
                                        <div class="mb-3">
                                            <label class="form-label">グラデーション方向</label>
                                            <select class="form-select" id="gradientDirection">
                                                <option value="135deg">右下へ（135度）</option>
                                                <option value="90deg">右へ（90度）</option>
                                                <option value="180deg">下へ（180度）</option>
                                                <option value="45deg">右上へ（45度）</option>
                                                <option value="0deg">上へ（0度）</option>
                                            </select>
                                        </div>
                                    </div>
                                    
                                    <!-- 画像設定 -->
                                    <div id="imageSettings" class="header-type-settings" style="display: none;">
                                        <h6 class="mt-4">画像設定</h6>
                                        <div class="mb-3">
                                            <label class="form-label">ヘッダー画像</label>
                                            <input type="file" class="form-control" id="headerImage" accept="image/*">
                                            <small class="text-muted">推奨: 横幅1920px以上、高さ600px以上の画像</small>
                                        </div>
                                        <div class="mb-3">
                                            <label class="form-label">画像の表示方法</label>
                                            <select class="form-select" id="headerImageDisplay">
                                                <option value="cover">カバー（画面に合わせて拡大・縮小）</option>
                                                <option value="contain">全体表示（アスペクト比を維持）</option>
                                                <option value="fixed">固定背景</option>
                                            </select>
                                        </div>
                                        <div class="mb-3">
                                            <label class="form-label">オーバーレイ設定</label>
                                            <select class="form-select" id="headerImageOverlay">
                                                <option value="none">なし</option>
                                                <option value="dark">暗いオーバーレイ</option>
                                                <option value="light">明るいオーバーレイ</option>
                                                <option value="gradient">グラデーションオーバーレイ</option>
                                            </select>
                                        </div>
                                    </div>
                                    
                                    <!-- パターン設定 -->
                                    <div id="patternSettings" class="header-type-settings" style="display: none;">
                                        <h6 class="mt-4">パターン設定</h6>
                                        <div class="row">
                                            <div class="col-md-6 mb-3">
                                                <label class="form-label">パターンカラー</label>
                                                <input type="color" class="form-control form-control-color" id="patternColor" value="#667eea">
                                            </div>
                                            <div class="col-md-6 mb-3">
                                                <label class="form-label">背景カラー</label>
                                                <input type="color" class="form-control form-control-color" id="patternBackground" value="#1a1a2e">
                                            </div>
                                        </div>
                                        <div class="mb-3">
                                            <label class="form-label">アニメーション速度</label>
                                            <select class="form-select" id="patternSpeed">
                                                <option value="slow">ゆっくり</option>
                                                <option value="normal" selected>通常</option>
                                                <option value="fast">速い</option>
                                            </select>
                                        </div>
                                    </div>
                                    
                                    <!-- 共通設定 -->
                                    <h6 class="mt-4">共通設定</h6>
                                    <div class="row">
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label">ヘッダーの高さ</label>
                                            <select class="form-select" id="headerHeight">
                                                <option value="small">小（50vh）</option>
                                                <option value="medium" selected>中（70vh）</option>
                                                <option value="large">大（90vh）</option>
                                                <option value="full">全画面（100vh）</option>
                                            </select>
                                        </div>
                                        <div class="col-md-6 mb-3">
                                            <label class="form-label">テキストカラー</label>
                                            <input type="color" class="form-control form-control-color" id="headerTextColor" value="#ffffff">
                                        </div>
                                    </div>
                                    
                                    <!-- プレビュー -->
                                    <div class="card mt-4">
                                        <div class="card-header">プレビュー</div>
                                        <div class="card-body p-0">
                                            <div id="headerPreview" style="height: 300px; position: relative; overflow: hidden;">
                                                <div class="header-preview-content" style="position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); text-align: center; z-index: 10;">
                                                    <h1 style="color: #fff; margin-bottom: 10px;">サイトタイトル</h1>
                                                    <p style="color: #fff; opacity: 0.9;">キャッチフレーズのサンプルテキスト</p>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- 見出しデザインタブ -->
                        <div class="tab-pane fade" id="headingTab" role="tabpanel">
                            <div class="row">
                                <div class="col-md-4 mb-4">
                                    <h6>H1見出し設定</h6>
                                    <div class="mb-2">
                                        <label class="form-label">デザインスタイル</label>
                                        <select class="form-select" id="h1Style">
                                            <option value="none">デザイン無し</option>
                                            <option value="filled">塗りつぶし帯</option>
                                            <option value="underline">下線</option>
                                            <option value="checkmark">チェックマーク</option>
                                        </select>
                                    </div>
                                    <div class="mb-2">
                                        <label class="form-label">カラー</label>
                                        <input type="color" class="form-control form-control-color" id="h1Color" value="#333333">
                                    </div>
                                </div>
                                
                                <div class="col-md-4 mb-4">
                                    <h6>H2見出し設定</h6>
                                    <div class="mb-2">
                                        <label class="form-label">デザインスタイル</label>
                                        <select class="form-select" id="h2Style">
                                            <option value="none">デザイン無し</option>
                                            <option value="filled">塗りつぶし帯</option>
                                            <option value="underline">下線</option>
                                            <option value="checkmark">チェックマーク</option>
                                        </select>
                                    </div>
                                    <div class="mb-2">
                                        <label class="form-label">カラー</label>
                                        <input type="color" class="form-control form-control-color" id="h2Color" value="#555555">
                                    </div>
                                </div>
                                
                                <div class="col-md-4 mb-4">
                                    <h6>H3見出し設定</h6>
                                    <div class="mb-2">
                                        <label class="form-label">デザインスタイル</label>
                                        <select class="form-select" id="h3Style">
                                            <option value="none">デザイン無し</option>
                                            <option value="filled">塗りつぶし帯</option>
                                            <option value="underline">下線</option>
                                            <option value="checkmark">チェックマーク</option>
                                        </select>
                                    </div>
                                    <div class="mb-2">
                                        <label class="form-label">カラー</label>
                                        <input type="color" class="form-control form-control-color" id="h3Color" value="#777777">
                                    </div>
                                </div>
                                
                                <div class="col-md-4 mb-4">
                                    <h6>H4見出し設定</h6>
                                    <div class="mb-2">
                                        <label class="form-label">デザインスタイル</label>
                                        <select class="form-select" id="h4Style">
                                            <option value="none">デザイン無し</option>
                                            <option value="filled">塗りつぶし帯</option>
                                            <option value="underline">下線</option>
                                            <option value="checkmark">チェックマーク</option>
                                        </select>
                                    </div>
                                    <div class="mb-2">
                                        <label class="form-label">カラー</label>
                                        <input type="color" class="form-control form-control-color" id="h4Color" value="#999999">
                                    </div>
                                </div>
                            </div>
                            
                            <!-- プレビューエリア -->
                            <div class="card mt-3">
                                <div class="card-header">プレビュー</div>
                                <div class="card-body" id="headingPreview">
                                    <h1 class="preview-h1">H1見出しのサンプル</h1>
                                    <h2 class="preview-h2">H2見出しのサンプル</h2>
                                    <h3 class="preview-h3">H3見出しのサンプル</h3>
                                    <h4 class="preview-h4">H4見出しのサンプル</h4>
                                </div>
                            </div>
                        </div>
                        
                        <!-- リストデザインタブ -->
                        <div class="tab-pane fade" id="listStyleTab" role="tabpanel">
                            <div class="row">
                                <div class="col-md-6">
                                    <h6 class="mb-3">リストマーカースタイル</h6>
                                    
                                    <!-- 順序なしリスト（UL）設定 -->
                                    <div class="mb-4">
                                        <label class="form-label">順序なしリスト（箇条書き）</label>
                                        <select class="form-select mb-2" id="ulStyle" name="list_styles[ul_style]">
                                            <option value="default">標準（・）</option>
                                            <option value="check">チェックマーク（✓）</option>
                                            <option value="arrow">矢印（➜）</option>
                                            <option value="star">スター（★）</option>
                                            <option value="heart">ハート（♥）</option>
                                            <option value="circle">サークル（●）</option>
                                            <option value="square">スクエア（■）</option>
                                            <option value="diamond">ダイヤモンド（◆）</option>
                                            <option value="hand">ハンドポイント（☞）</option>
                                            <option value="custom">カスタム</option>
                                        </select>
                                        
                                        <div class="input-group mb-2" id="ulColorGroup">
                                            <span class="input-group-text">色</span>
                                            <input type="color" class="form-control form-control-color" 
                                                   id="ulColor" name="list_styles[ul_color]" value="#333333">
                                        </div>
                                        
                                        <input type="text" class="form-control mb-2" id="ulCustomChar" 
                                               name="list_styles[ul_custom_char]" placeholder="カスタム文字"
                                               style="display: none;">
                                    </div>
                                    
                                    <!-- 順序付きリスト（OL）設定 -->
                                    <div class="mb-4">
                                        <label class="form-label">順序付きリスト（番号付き）</label>
                                        <select class="form-select mb-2" id="olStyle" name="list_styles[ol_style]">
                                            <option value="default">標準（1, 2, 3...）</option>
                                            <option value="circle">円形番号</option>
                                            <option value="square">四角番号</option>
                                            <option value="roman">ローマ数字（I, II, III...）</option>
                                            <option value="alpha">アルファベット（A, B, C...）</option>
                                        </select>
                                        
                                        <div class="input-group mb-2">
                                            <span class="input-group-text">色</span>
                                            <input type="color" class="form-control form-control-color" 
                                                   id="olColor" name="list_styles[ol_color]" value="#333333">
                                        </div>
                                    </div>
                                    
                                    <!-- アニメーション設定 -->
                                    <div class="mb-4">
                                        <div class="form-check form-switch">
                                            <input class="form-check-input" type="checkbox" 
                                                   id="listAnimation" name="list_styles[animation_enabled]">
                                            <label class="form-check-label" for="listAnimation">
                                                リストアニメーションを有効にする
                                            </label>
                                        </div>
                                        
                                        <select class="form-select mt-2" id="listAnimationType" 
                                                name="list_styles[animation_type]" style="display: none;">
                                            <option value="fade">フェードイン</option>
                                            <option value="slide">スライドイン</option>
                                            <option value="bounce">バウンス</option>
                                            <option value="rotate">回転</option>
                                        </select>
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <h6 class="mb-3">プレビュー</h6>
                                    <div class="border rounded p-3" id="listStylePreview" style="background: #f8f9fa;">
                                        <h5>順序なしリスト</h5>
                                        <ul class="preview-ul">
                                            <li>項目1のサンプルテキスト</li>
                                            <li>項目2のサンプルテキスト</li>
                                            <li>項目3のサンプルテキスト</li>
                                        </ul>
                                        
                                        <h5 class="mt-3">順序付きリスト</h5>
                                        <ol class="preview-ol">
                                            <li>ステップ1のサンプル</li>
                                            <li>ステップ2のサンプル</li>
                                            <li>ステップ3のサンプル</li>
                                        </ol>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- 背景設定タブ -->
                        <div class="tab-pane fade" id="backgroundTab" role="tabpanel">
                            <div class="row">
                                <div class="col-md-6">
                                    <h6>ページ背景設定</h6>
                                    
                                    <!-- 背景タイプ選択 -->
                                    <div class="mb-3">
                                        <label class="form-label">背景タイプ</label>
                                        <select class="form-select" id="backgroundType" name="backgrounds[type]">
                                            <option value="normal">通常（単色・画像）</option>
                                            <option value="animated">アニメーション背景</option>
                                        </select>
                                    </div>
                                    
                                    <!-- 通常背景設定 -->
                                    <div id="normalBackgroundSettings">
                                        <div class="mb-3">
                                            <label class="form-label">背景色</label>
                                            <input type="color" class="form-control form-control-color" id="pageBackgroundColor" value="#ffffff">
                                        </div>
                                    </div>
                                    
                                    <!-- アニメーション背景設定 -->
                                    <div id="animatedBackgroundSettings" style="display: none;">
                                        <div class="mb-3">
                                            <label class="form-label">アニメーションスタイル</label>
                                            <select class="form-select" id="animationStyle" name="backgrounds[animation_style]">
                                                <option value="none">なし</option>
                                                <option value="aurora">オーロラ</option>
                                                <option value="gradient-shift">グラデーションシフト</option>
                                                <option value="wave">ウェーブ</option>
                                                <option value="particles">パーティクル</option>
                                                <option value="floating-bubbles">フローティングバブル</option>
                                                <option value="geometric-mesh">ジオメトリックメッシュ</option>
                                                <option value="color-pulse">カラーパルス</option>
                                                <option value="starfield">スターフィールド</option>
                                                <option value="liquid-gradient">リキッドグラデーション</option>
                                                <option value="matrix-rain">マトリックスレイン</option>
                                                <option value="cosmic-dust">コズミックダスト</option>
                                                <option value="neon-pulse">ネオンパルス</option>
                                                <option value="crystal-grid">クリスタルグリッド</option>
                                                <option value="galaxy-spiral">ギャラクシースパイラル</option>
                                                <option value="cyber-rain">サイバーレイン</option>
                                                <option value="holographic-wave">ホログラフィックウェーブ</option>
                                                <option value="plasma-storm">プラズマストーム</option>
                                                <option value="quantum-field">クォンタムフィールド</option>
                                                <option value="nebula-cloud">ネビュラクラウド</option>
                                            </select>
                                        </div>
                                        <div class="mb-3">
                                            <label class="form-label">アニメーションカラー</label>
                                            <div class="row g-2">
                                                <div class="col">
                                                    <input type="color" class="form-control form-control-color" 
                                                           id="animColor1" name="backgrounds[anim_color1]" value="#667eea">
                                                    <small class="text-muted">プライマリ</small>
                                                </div>
                                                <div class="col">
                                                    <input type="color" class="form-control form-control-color" 
                                                           id="animColor2" name="backgrounds[anim_color2]" value="#764ba2">
                                                    <small class="text-muted">セカンダリ</small>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="mb-3">
                                            <label class="form-label">アニメーション速度</label>
                                            <input type="range" class="form-range" id="animSpeed" 
                                                   name="backgrounds[anim_speed]" min="1" max="10" value="5">
                                            <small class="text-muted">遅い ← → 速い</small>
                                        </div>
                                    </div>
                                    <div class="mb-3">
                                        <label class="form-label">背景画像</label>
                                        <input type="file" class="form-control" id="pageBackgroundImage" accept="image/*">
                                        <small class="text-muted">推奨: JPG/PNG形式、最大2MB</small>
                                    </div>
                                    <div class="mb-3">
                                        <label class="form-label">背景画像の設定</label>
                                        <select class="form-select" id="pageBackgroundRepeat">
                                            <option value="no-repeat">繰り返しなし</option>
                                            <option value="repeat">繰り返し</option>
                                            <option value="repeat-x">横方向に繰り返し</option>
                                            <option value="repeat-y">縦方向に繰り返し</option>
                                        </select>
                                    </div>
                                    <div class="mb-3">
                                        <label class="form-label">背景画像の位置</label>
                                        <select class="form-select" id="pageBackgroundPosition">
                                            <option value="center center">中央</option>
                                            <option value="top left">左上</option>
                                            <option value="top center">上中央</option>
                                            <option value="top right">右上</option>
                                            <option value="center left">左中央</option>
                                            <option value="center right">右中央</option>
                                            <option value="bottom left">左下</option>
                                            <option value="bottom center">下中央</option>
                                            <option value="bottom right">右下</option>
                                        </select>
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <h6>背景プレビュー</h6>
                                    <div id="backgroundPreview" class="border rounded p-4 mb-3" style="height: 300px; position: relative; overflow: hidden;">
                                        <div id="animatedBgPreview" style="position: absolute; top: 0; left: 0; width: 100%; height: 100%; display: none;"></div>
                                        <div class="text-center" style="position: relative; z-index: 1;">
                                            <h5>サンプルページ</h5>
                                            <p>背景のプレビューがここに表示されます</p>
                                            <div class="card" style="max-width: 300px; margin: 0 auto;">
                                                <div class="card-body">
                                                    <h6>記事カード</h6>
                                                    <p>サンプルテキスト</p>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <h6>記事背景設定</h6>
                                    <div class="mb-3">
                                        <label class="form-label">背景色</label>
                                        <input type="color" class="form-control form-control-color" id="articleBackgroundColor" value="#ffffff">
                                    </div>
                                    <div class="mb-3">
                                        <label class="form-label">背景画像</label>
                                        <input type="file" class="form-control" id="articleBackgroundImage" accept="image/*">
                                        <small class="text-muted">推奨: JPG/PNG形式、最大2MB</small>
                                    </div>
                                    <div class="mb-3">
                                        <label class="form-label">背景画像の設定</label>
                                        <select class="form-select" id="articleBackgroundRepeat">
                                            <option value="no-repeat">繰り返しなし</option>
                                            <option value="repeat">繰り返し</option>
                                            <option value="repeat-x">横方向に繰り返し</option>
                                            <option value="repeat-y">縦方向に繰り返し</option>
                                        </select>
                                    </div>
                                    <div class="mb-3">
                                        <label class="form-label">背景画像の位置</label>
                                        <select class="form-select" id="articleBackgroundPosition">
                                            <option value="center center">中央</option>
                                            <option value="top left">左上</option>
                                            <option value="top center">上中央</option>
                                            <option value="top right">右上</option>
                                            <option value="center left">左中央</option>
                                            <option value="center right">右中央</option>
                                            <option value="bottom left">左下</option>
                                            <option value="bottom center">下中央</option>
                                            <option value="bottom right">右下</option>
                                        </select>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- テーマカラータブ -->
                        <div class="tab-pane fade" id="themeColorTab" role="tabpanel">
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label class="form-label">プライマリーカラー（メインテーマ色）</label>
                                        <input type="color" class="form-control form-control-color" id="primaryColor" value="#007bff">
                                        <small class="text-muted">ヘッダー、ボタン、リンクなどに使用されます</small>
                                    </div>
                                    <div class="mb-3">
                                        <label class="form-label">セカンダリーカラー（サブテーマ色）</label>
                                        <input type="color" class="form-control form-control-color" id="secondaryColor" value="#6c757d">
                                        <small class="text-muted">補助的な要素に使用されます</small>
                                    </div>
                                    <div class="mb-3">
                                        <label class="form-label">アクセントカラー</label>
                                        <input type="color" class="form-control form-control-color" id="accentColor" value="#28a745">
                                        <small class="text-muted">重要な要素を強調する際に使用されます</small>
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label class="form-label">テキストカラー</label>
                                        <input type="color" class="form-control form-control-color" id="textColor" value="#333333">
                                        <small class="text-muted">本文のテキスト色</small>
                                    </div>
                                    <div class="mb-3">
                                        <label class="form-label">リンクカラー</label>
                                        <input type="color" class="form-control form-control-color" id="linkColor" value="#007bff">
                                        <small class="text-muted">通常のリンク色</small>
                                    </div>
                                    <div class="mb-3">
                                        <label class="form-label">リンクホバーカラー</label>
                                        <input type="color" class="form-control form-control-color" id="linkHoverColor" value="#0056b3">
                                        <small class="text-muted">マウスオーバー時のリンク色</small>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- カラープレビュー -->
                            <div class="card mt-3">
                                <div class="card-header">カラープレビュー</div>
                                <div class="card-body" id="colorPreview">
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="p-3 mb-2 text-white" id="primaryPreview">プライマリーカラー</div>
                                            <div class="p-3 mb-2 text-white" id="secondaryPreview">セカンダリーカラー</div>
                                            <div class="p-3 mb-2 text-white" id="accentPreview">アクセントカラー</div>
                                        </div>
                                        <div class="col-md-6">
                                            <p id="textPreview">これはサンプルテキストです。</p>
                                            <p><a href="#" id="linkPreview">これはサンプルリンクです</a></p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- 検証・トラッキングタブ -->
                        <div class="tab-pane fade" id="verificationTab" role="tabpanel">
                            <div class="row">
                                <div class="col-md-6">
                                    <h6><i class="bi bi-google"></i> Google関連</h6>
                                    <div class="mb-3">
                                        <label class="form-label">Google Search Console 検証コード</label>
                                        <textarea class="form-control" id="googleSearchConsole" rows="3" placeholder='<meta name="google-site-verification" content="..." />'></textarea>
                                        <small class="text-muted">Google Search Consoleから提供されるmetaタグ全体を貼り付けてください</small>
                                    </div>
                                    <div class="mb-3">
                                        <label class="form-label">Google Analytics トラッキングコード</label>
                                        <textarea class="form-control" id="googleAnalytics" rows="6" placeholder="<!-- Google Analytics -->
<script>
(function(i,s,o,g,r,a,m){...
</script>"></textarea>
                                        <small class="text-muted">Google Analyticsのトラッキングコード全体（scriptタグを含む）を貼り付けてください</small>
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <h6><i class="bi bi-facebook"></i> その他のサービス</h6>
                                    <div class="mb-3">
                                        <label class="form-label">Facebook Pixel コード</label>
                                        <textarea class="form-control" id="facebookPixel" rows="6" placeholder="<!-- Facebook Pixel Code -->
<script>
!function(f,b,e,v,n,t,s){...
</script>"></textarea>
                                        <small class="text-muted">Facebook Pixelのコード全体を貼り付けてください</small>
                                    </div>
                                    <div class="mb-3">
                                        <label class="form-label">カスタムヘッドコード</label>
                                        <textarea class="form-control" id="customHeadCode" rows="4" placeholder="その他の検証コードやメタタグなど"></textarea>
                                        <small class="text-muted">その他の検証コードやカスタムメタタグを追加できます（&lt;head&gt;内に挿入されます）</small>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="alert alert-info mt-3">
                                <i class="bi bi-info-circle"></i> これらのコードは生成されるHTMLの適切な位置に自動的に挿入されます。
                                <ul class="mb-0 mt-2">
                                    <li>Google Search Console、カスタムヘッドコード: &lt;head&gt;タグ内</li>
                                    <li>Google Analytics: &lt;/head&gt;タグの直前</li>
                                    <li>Facebook Pixel: &lt;/body&gt;タグの直前</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">キャンセル</button>
                <button type="button" class="btn btn-primary" id="saveDesignSettings">
                    <i class="bi bi-save"></i> 設定を保存
                </button>
            </div>
        </div>
    </div>
</div>

<!-- 更新プログレスモーダル -->
<div class="modal fade" id="updateProgressModal" tabindex="-1" data-bs-backdrop="static">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">サイト更新中...</h5>
            </div>
            <div class="modal-body">
                <div class="progress mb-3">
                    <div class="progress-bar progress-bar-striped progress-bar-animated" 
                         role="progressbar" style="width: 0%" id="updateProgress"></div>
                </div>
                <div id="updateLog" class="small text-muted" style="max-height: 200px; overflow-y: auto;"></div>
            </div>
        </div>
    </div>
</div>

<script>
// PHPの記事データをJavaScriptで使用
const allArticles = <?php echo json_encode($articles); ?>;
const siteConfigs = <?php echo json_encode($siteConfigs); ?>;

$(document).ready(function() {
    // FTPプロファイル選択ボタン
    $('.select-ftp-btn').click(function() {
        const configId = $(this).data('config-id');
        const siteName = $(this).data('site-name');
        
        $('#ftpSelectTitle').text(`${siteName} - FTP設定を選択`);
        window.currentFtpConfigId = configId;
        
        const modal = new bootstrap.Modal(document.getElementById('ftpSelectModal'));
        modal.show();
    });
    
    // FTPプロファイル選択
    $('.ftp-profile-item').click(function() {
        const profileId = $(this).data('profile-id');
        const configId = window.currentFtpConfigId;
        
        // APIでFTPプロファイルを保存
        $.ajax({
            url: '../api/update-site-ftp.php',
            method: 'POST',
            data: {
                config_id: configId,
                ftp_profile_id: profileId
            },
            success: function(response) {
                if (response.success) {
                    // モーダルを閉じる
                    bootstrap.Modal.getInstance(document.getElementById('ftpSelectModal')).hide();
                    
                    // 成功メッセージを表示
                    const statusDiv = $(`#status-${configId}`);
                    statusDiv.html(`
                        <div class="alert alert-success alert-dismissible fade show" role="alert">
                            <i class="bi bi-check-circle"></i> FTP設定を更新しました
                            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                        </div>
                    `);
                    
                    // 3秒後にページをリロード
                    setTimeout(() => {
                        location.reload();
                    }, 2000);
                } else {
                    alert('エラー: ' + response.error);
                }
            },
            error: function() {
                alert('通信エラーが発生しました');
            }
        });
    });
    
    // 記事管理ボタン
    $('.manage-articles-btn').click(function() {
        const configId = $(this).data('config-id');
        const siteName = $(this).data('site-name');
        const siteConfig = siteConfigs[configId];
        
        // モーダルタイトル設定
        $('#articleManageTitle').text(`${siteName} - 記事の管理`);
        
        // 現在の選択状態を保存
        window.currentConfigId = configId;
        window.currentArticleIds = siteConfig.article_ids || [];
        
        // 記事リストを生成
        renderArticleList();
        
        // モーダル表示
        const modal = new bootstrap.Modal(document.getElementById('articleManageModal'));
        modal.show();
    });
    
    // 記事リストのレンダリング
    function renderArticleList() {
        const container = $('#modalArticleList');
        container.empty();
        
        // 記事を逆順（新しい順）で表示
        allArticles.slice().reverse().forEach((article, index) => {
            const realIndex = allArticles.length - 1 - index;
            const articleId = article.id || `idx_${realIndex}`;
            const isChecked = window.currentArticleIds.includes(articleId);
            
            const articleItem = $(`
                <div class="form-check article-item-modal" data-category="${article.category || ''}">
                    <input type="checkbox" class="form-check-input article-checkbox-modal" 
                           value="${articleId}" 
                           id="modal_article_${realIndex}"
                           ${isChecked ? 'checked' : ''}>
                    <label class="form-check-label" for="modal_article_${realIndex}">
                        <strong>${escapeHtml(article.title)}</strong>
                        <small class="text-muted">
                            - ${article.category || 'その他'}
                            (${new Date(article.created_at).toLocaleDateString('ja-JP')})
                        </small>
                    </label>
                </div>
            `);
            
            container.append(articleItem);
        });
        
        updateModalSelectedCount();
    }
    
    // カテゴリーフィルター（モーダル）
    $('#modalCategoryFilter').change(function() {
        const category = $(this).val();
        if (category === '') {
            $('.article-item-modal').show();
        } else {
            $('.article-item-modal').hide();
            $(`.article-item-modal[data-category="${category}"]`).show();
        }
    });
    
    // 全選択/解除（モーダル）
    $('#modalSelectAll').change(function() {
        $('.article-checkbox-modal:visible').prop('checked', $(this).prop('checked'));
        updateModalSelectedCount();
    });
    
    // メタディスクリプションの文字数カウント
    $('#siteMetaDescription').on('input', function() {
        updateMetaDescLength();
    });
    
    function updateMetaDescLength() {
        const length = $('#siteMetaDescription').val().length;
        $('#metaDescLength').text(length);
        
        // 推奨文字数に応じて色を変更
        if (length > 120) {
            $('#metaDescLength').removeClass('text-primary').addClass('text-warning');
        } else {
            $('#metaDescLength').removeClass('text-warning').addClass('text-primary');
        }
    }
    
    // 選択数を更新（モーダル）
    $(document).on('change', '.article-checkbox-modal', updateModalSelectedCount);
    
    function updateModalSelectedCount() {
        const count = $('.article-checkbox-modal:checked').length;
        $('#modalSelectedCount').text(count);
    }
    
    // 記事選択を保存
    $('#saveArticleSelection').click(function() {
        const selectedIds = [];
        $('.article-checkbox-modal:checked').each(function() {
            selectedIds.push($(this).val());
        });
        
        // APIで設定を更新
        $.ajax({
            url: '../api/update-site-articles.php',
            method: 'POST',
            data: {
                config_id: window.currentConfigId,
                article_ids: selectedIds
            },
            success: function(response) {
                if (response.success) {
                    // 成功メッセージ
                    alert('記事の選択を保存しました');
                    
                    // ローカルデータを更新
                    siteConfigs[window.currentConfigId].article_ids = selectedIds;
                    
                    // 記事数表示を更新
                    location.reload(); // 簡単のためリロード
                } else {
                    alert('エラー: ' + response.error);
                }
            },
            error: function() {
                alert('通信エラーが発生しました');
            }
        });
    });
    
    // HTMLエスケープ関数
    function escapeHtml(text) {
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.replace(/[&<>"']/g, m => map[m]);
    }
    
    // サイト更新のみ
    $('.update-site-btn').click(function() {
        const configId = $(this).data('config-id');
        const siteName = $(this).data('site-name');
        
        if (!confirm(`「${siteName}」を更新しますか？`)) {
            return;
        }
        
        updateSite(configId, false);
    });
    
    // 更新してデプロイ
    $('.update-and-deploy-btn').click(function() {
        const configId = $(this).data('config-id');
        const siteName = $(this).data('site-name');
        
        if (!confirm(`「${siteName}」を更新してFTPでデプロイしますか？`)) {
            return;
        }
        
        updateSite(configId, true);
    });
    
    // サイト更新処理
    function updateSite(configId, deploy = false) {
        const statusDiv = $(`#status-${configId}`);
        const modal = new bootstrap.Modal(document.getElementById('updateProgressModal'));
        const progressBar = $('#updateProgress');
        const updateLog = $('#updateLog');
        
        // モーダル表示
        modal.show();
        progressBar.css('width', '0%');
        updateLog.empty();
        
        // ログ追加関数
        function addLog(message, type = 'info') {
            const time = new Date().toLocaleTimeString('ja-JP');
            const color = type === 'error' ? 'text-danger' : (type === 'success' ? 'text-success' : 'text-muted');
            updateLog.append(`<div class="${color}">[${time}] ${message}</div>`);
            updateLog.scrollTop(updateLog[0].scrollHeight);
        }
        
        addLog('更新処理を開始します...');
        progressBar.css('width', '10%');
        
        // サイト設定を読み込んで更新
        $.ajax({
            url: '../api/update-site.php',
            method: 'POST',
            data: { 
                config_id: configId,
                action: 'update',
                deploy: deploy
            },
            success: function(response) {
                progressBar.css('width', '50%');
                
                if (response.success) {
                    addLog('サイト生成完了', 'success');
                    addLog(`生成ファイル数: ${response.files.length}`);
                    
                    if (deploy && response.deploy_result) {
                        progressBar.css('width', '80%');
                        addLog('FTPデプロイを開始...');
                        
                        if (response.deploy_result.success) {
                            progressBar.css('width', '100%');
                            addLog(`デプロイ完了: ${response.deploy_result.uploaded} ファイル`, 'success');
                            
                            statusDiv.html(`
                                <div class="alert alert-success">
                                    <i class="bi bi-check-circle"></i> 更新とデプロイが完了しました！
                                    <br><small>${new Date().toLocaleString('ja-JP')}</small>
                                </div>
                            `);
                        } else {
                            addLog('デプロイエラー: ' + response.deploy_result.error, 'error');
                            statusDiv.html(`
                                <div class="alert alert-warning">
                                    <i class="bi bi-exclamation-triangle"></i> サイト生成は完了しましたが、デプロイに失敗しました。
                                </div>
                            `);
                        }
                    } else {
                        progressBar.css('width', '100%');
                        statusDiv.html(`
                            <div class="alert alert-success">
                                <i class="bi bi-check-circle"></i> サイト更新が完了しました！
                                <br><small>${new Date().toLocaleString('ja-JP')}</small>
                            </div>
                        `);
                    }
                    
                    // 3秒後にモーダルを閉じる
                    setTimeout(() => {
                        modal.hide();
                    }, 3000);
                    
                } else {
                    addLog('エラー: ' + response.error, 'error');
                    statusDiv.html(`
                        <div class="alert alert-danger">
                            <i class="bi bi-x-circle"></i> 更新エラー: ${response.error}
                        </div>
                    `);
                    modal.hide();
                }
            },
            error: function(xhr, status, error) {
                addLog('通信エラー: ' + error, 'error');
                statusDiv.html(`
                    <div class="alert alert-danger">
                        <i class="bi bi-x-circle"></i> 通信エラーが発生しました。
                    </div>
                `);
                modal.hide();
            }
        });
    }
    
    // 設定編集（サイト生成ページへ遷移）
    $('.edit-config-btn').click(function() {
        const configId = $(this).data('config-id');
        window.location.href = `?page=site-generator&edit=${configId}`;
    });
    
    // 設定削除
    $('.delete-config-btn').click(function() {
        const configId = $(this).data('config-id');
        const siteName = $(this).data('site-name');
        
        if (!confirm(`「${siteName}」の設定を削除しますか？\n※生成されたサイトファイルは削除されません`)) {
            return;
        }
        
        $.ajax({
            url: '../api/delete-site-config.php',
            method: 'POST',
            data: { config_id: configId },
            success: function(response) {
                if (response.success) {
                    alert('設定を削除しました');
                    location.reload();
                } else {
                    alert('削除エラー: ' + response.error);
                }
            },
            error: function() {
                alert('通信エラーが発生しました');
            }
        });
    });
    
    // デザイン設定ボタン
    $('.design-settings-btn').click(function() {
        const configId = $(this).data('config-id');
        const siteName = $(this).data('site-name');
        const siteConfig = siteConfigs[configId];
        
        // モーダルタイトル設定
        $('#designSettingsTitle').text(`${siteName} - デザイン設定`);
        
        // 現在の設定IDを保存
        window.currentDesignConfigId = configId;
        
        // 既存の設定を読み込み
        if (siteConfig.design_settings) {
            loadDesignSettings(siteConfig.design_settings);
        } else {
            resetDesignSettings();
            // 検証コードが別の場所に保存されている場合も確認
            if (siteConfig.verification_codes) {
                $('#googleSearchConsole').val(siteConfig.verification_codes.google_search_console || '');
                $('#googleAnalytics').val(siteConfig.verification_codes.google_analytics || '');
                $('#facebookPixel').val(siteConfig.verification_codes.facebook_pixel || '');
                $('#customHeadCode').val(siteConfig.verification_codes.custom_head_code || '');
            }
        }
        
        // TinyMCEの初期化
        initTinyMCE();
        
        // ヘッダープレビューの初期化
        updateHeaderPreview();
        
        // モーダル表示
        const modal = new bootstrap.Modal(document.getElementById('designSettingsModal'));
        modal.show();
    });
    
    // デザイン設定の読み込み
    function loadDesignSettings(settings) {
        // 基本情報の読み込み（siteConfigから）
        const siteConfig = siteConfigs[window.currentDesignConfigId];
        if (siteConfig) {
            $('#siteDisplayName').val(siteConfig.display_name || '');
            $('#siteDescription').val(siteConfig.description || '');
            $('#heroCatchphrase').val(siteConfig.hero_catchphrase || '');
            $('#siteKeywords').val(siteConfig.seo_keywords || '');
            $('#siteMetaDescription').val(siteConfig.meta_description || '');
            $('#ogpTitle').val(siteConfig.ogp_title || '');
            $('#ogpDescription').val(siteConfig.ogp_description || '');
            
            // メタディスクリプションの文字数更新
            updateMetaDescLength();
        }
        
        // レイアウト設定
        if (settings.template) {
            $('#columnLayout').val(settings.template);
        } else {
            // デフォルトは2カラム
            $('#columnLayout').val('modern-blog');
        }
        
        // 記事表示件数の読み込み
        if (siteConfig && siteConfig.articles_per_page) {
            $('#articlesPerPage').val(siteConfig.articles_per_page);
        } else {
            $('#articlesPerPage').val(10); // デフォルト値
        }
        
        // フリースペース
        if (settings.free_spaces) {
            $('#topUpperFreeSpace').val(settings.free_spaces.top_upper || '');
            $('#topLowerFreeSpace').val(settings.free_spaces.top_lower || '');
            $('#articleUpperFreeSpace').val(settings.free_spaces.article_upper || '');
            $('#articleLowerFreeSpace').val(settings.free_spaces.article_lower || '');
        } else if (siteConfig) {
            // 基本設定のフリースペースデータがある場合はそれを使用
            $('#topUpperFreeSpace').val(siteConfig.top_free_space || '');
            $('#topLowerFreeSpace').val(siteConfig.bottom_free_space || '');
            $('#articleUpperFreeSpace').val(''); // 基本設定には上部がないため空
            $('#articleLowerFreeSpace').val(siteConfig.article_free_space || '');
        }
        
        // ヘッダーデザイン
        if (settings.header_design) {
            $('#headerType').val(settings.header_design.type || 'gradient');
            $('#headerHeight').val(settings.header_design.height || 'medium');
            $('#headerTextColor').val(settings.header_design.text_color || '#ffffff');
            
            // グラデーション設定
            if (settings.header_design.gradient) {
                $('#gradientStart').val(settings.header_design.gradient.start || '#667eea');
                $('#gradientEnd').val(settings.header_design.gradient.end || '#764ba2');
                $('#gradientDirection').val(settings.header_design.gradient.direction || '135deg');
            }
            
            // 画像設定
            if (settings.header_design.image) {
                $('#headerImageDisplay').val(settings.header_design.image.display || 'cover');
                $('#headerImageOverlay').val(settings.header_design.image.overlay || 'none');
            }
            
            // パターン設定
            if (settings.header_design.pattern) {
                $('#patternColor').val(settings.header_design.pattern.color || '#667eea');
                $('#patternBackground').val(settings.header_design.pattern.background || '#1a1a2e');
                $('#patternSpeed').val(settings.header_design.pattern.speed || 'normal');
            }
            
            // 設定パネルの表示切り替え
            toggleHeaderSettings($('#headerType').val());
        }
        
        // 見出しデザイン
        if (settings.heading_styles) {
            $('#h1Style').val(settings.heading_styles.h1_style || 'none');
            $('#h1Color').val(settings.heading_styles.h1_color || '#333333');
            $('#h2Style').val(settings.heading_styles.h2_style || 'none');
            $('#h2Color').val(settings.heading_styles.h2_color || '#555555');
            $('#h3Style').val(settings.heading_styles.h3_style || 'none');
            $('#h3Color').val(settings.heading_styles.h3_color || '#777777');
            $('#h4Style').val(settings.heading_styles.h4_style || 'none');
            $('#h4Color').val(settings.heading_styles.h4_color || '#999999');
        }
        
        // リストスタイル
        if (settings.list_styles) {
            $('#ulStyle').val(settings.list_styles.ul_style || 'default');
            $('#ulColor').val(settings.list_styles.ul_color || '#333333');
            $('#ulCustomChar').val(settings.list_styles.ul_custom_char || '');
            $('#olStyle').val(settings.list_styles.ol_style || 'default');
            $('#olColor').val(settings.list_styles.ol_color || '#333333');
            $('#listAnimation').prop('checked', settings.list_styles.animation_enabled || false);
            $('#listAnimationType').val(settings.list_styles.animation_type || 'fade');
            
            // カスタム文字フィールドとアニメーションタイプの表示制御
            if ($('#ulStyle').val() === 'custom') {
                $('#ulCustomChar').show();
            } else {
                $('#ulCustomChar').hide();
            }
            
            if ($('#listAnimation').is(':checked')) {
                $('#listAnimationType').show();
            } else {
                $('#listAnimationType').hide();
            }
        }
        
        // 背景設定
        if (settings.backgrounds) {
            // 背景タイプの判定
            const hasAnimation = settings.backgrounds.animation_style && settings.backgrounds.animation_style !== 'none';
            $('#backgroundType').val(hasAnimation ? 'animated' : 'normal');
            
            // 通常背景設定
            $('#pageBackgroundColor').val(settings.backgrounds.page_bg_color || '#ffffff');
            $('#pageBackgroundRepeat').val(settings.backgrounds.page_bg_repeat || 'no-repeat');
            $('#pageBackgroundPosition').val(settings.backgrounds.page_bg_position || 'center center');
            $('#articleBackgroundColor').val(settings.backgrounds.article_bg_color || '#ffffff');
            $('#articleBackgroundRepeat').val(settings.backgrounds.article_bg_repeat || 'no-repeat');
            $('#articleBackgroundPosition').val(settings.backgrounds.article_bg_position || 'center center');
            
            // アニメーション背景設定
            if (hasAnimation) {
                $('#animationStyle').val(settings.backgrounds.animation_style);
                $('#animColor1').val(settings.backgrounds.anim_color1 || '#667eea');
                $('#animColor2').val(settings.backgrounds.anim_color2 || '#764ba2');
                $('#animSpeed').val(settings.backgrounds.anim_speed || 'normal');
                
                // アニメーション設定を表示
                $('#normalBackgroundSettings').hide();
                $('#animatedBackgroundSettings').show();
            } else {
                // アニメーションスタイルを「なし」にリセット
                $('#animationStyle').val('none');
                // 通常設定を表示
                $('#normalBackgroundSettings').show();
                $('#animatedBackgroundSettings').hide();
            }
        }
        
        // テーマカラー
        if (settings.theme_colors) {
            $('#primaryColor').val(settings.theme_colors.primary || '#007bff');
            $('#secondaryColor').val(settings.theme_colors.secondary || '#6c757d');
            $('#accentColor').val(settings.theme_colors.accent || '#28a745');
            $('#textColor').val(settings.theme_colors.text || '#333333');
            $('#linkColor').val(settings.theme_colors.link || '#007bff');
            $('#linkHoverColor').val(settings.theme_colors.link_hover || '#0056b3');
        }
        
        // 検証・トラッキングコード
        if (settings.verification_codes) {
            $('#googleSearchConsole').val(settings.verification_codes.google_search_console || '');
            $('#googleAnalytics').val(settings.verification_codes.google_analytics || '');
            $('#facebookPixel').val(settings.verification_codes.facebook_pixel || '');
            $('#customHeadCode').val(settings.verification_codes.custom_head_code || '');
        }
        
        // プレビュー更新
        updateHeadingPreview();
        updateColorPreview();
        updateListPreview();
    }
    
    // デザイン設定のリセット
    function resetDesignSettings() {
        // 基本情報はリセットせず、既存のデータを読み込む
        const siteConfig = siteConfigs[window.currentDesignConfigId];
        if (siteConfig) {
            $('#siteDisplayName').val(siteConfig.display_name || '');
            $('#siteDescription').val(siteConfig.description || '');
            $('#heroCatchphrase').val(siteConfig.hero_catchphrase || '');
            $('#siteKeywords').val(siteConfig.seo_keywords || '');
            $('#siteMetaDescription').val(siteConfig.meta_description || '');
            $('#ogpTitle').val(siteConfig.ogp_title || '');
            $('#ogpDescription').val(siteConfig.ogp_description || '');
            updateMetaDescLength();
        }
        
        $('#topUpperFreeSpace').val('');
        $('#topLowerFreeSpace').val('');
        $('#articleUpperFreeSpace').val('');
        $('#articleLowerFreeSpace').val('');
        
        $('#headerType').val('gradient');
        $('#headerHeight').val('medium');
        $('#headerTextColor').val('#ffffff');
        $('#gradientStart').val('#667eea');
        $('#gradientEnd').val('#764ba2');
        $('#gradientDirection').val('135deg');
        $('#headerImageDisplay').val('cover');
        $('#headerImageOverlay').val('none');
        $('#patternColor').val('#667eea');
        $('#patternBackground').val('#1a1a2e');
        $('#patternSpeed').val('normal');
        
        $('#h1Style').val('none');
        $('#h1Color').val('#333333');
        $('#h2Style').val('none');
        $('#h2Color').val('#555555');
        $('#h3Style').val('none');
        $('#h3Color').val('#777777');
        
        // リストスタイルのリセット
        $('#ulStyle').val('default');
        $('#ulColor').val('#333333');
        $('#ulCustomChar').val('').hide();
        $('#olStyle').val('default');
        $('#olColor').val('#333333');
        $('#listAnimation').prop('checked', false);
        $('#listAnimationType').val('fade').hide();
        
        $('#pageBackgroundColor').val('#ffffff');
        $('#articleBackgroundColor').val('#ffffff');
        
        $('#primaryColor').val('#007bff');
        $('#secondaryColor').val('#6c757d');
        $('#accentColor').val('#28a745');
        $('#textColor').val('#333333');
        $('#linkColor').val('#007bff');
        $('#linkHoverColor').val('#0056b3');
        
        // 検証コードはリセットしない（既存の値を保持）
        // $('#googleSearchConsole').val('');
        // $('#googleAnalytics').val('');
        // $('#facebookPixel').val('');
        // $('#customHeadCode').val('');
        
        toggleHeaderSettings('gradient');
        updateHeaderPreview();
        updateHeadingPreview();
        updateColorPreview();
    }
    
    // TinyMCEの初期化（無効化）
    function initTinyMCE() {
        // TinyMCEは使用しない（APIキーが必要なため）
        // 通常のtextareaを使用
    }
    
    // ヘッダータイプ変更時の処理
    $('#headerType').change(function() {
        toggleHeaderSettings($(this).val());
        updateHeaderPreview();
    });
    
    // ヘッダー設定パネルの表示切り替え
    function toggleHeaderSettings(type) {
        $('.header-type-settings').hide();
        
        if (type === 'gradient') {
            $('#gradientSettings').show();
        } else if (type === 'image') {
            $('#imageSettings').show();
        } else if (type.startsWith('pattern-')) {
            $('#patternSettings').show();
        }
    }
    
    // ヘッダープレビューの更新
    function updateHeaderPreview() {
        const type = $('#headerType').val();
        const textColor = $('#headerTextColor').val();
        const height = $('#headerHeight').val();
        const preview = $('#headerPreview');
        
        // 高さの設定を反映
        let previewHeight = '300px'; // デフォルト
        switch(height) {
            case 'small':
                previewHeight = '200px';
                break;
            case 'medium':
                previewHeight = '300px';
                break;
            case 'large':
                previewHeight = '400px';
                break;
            case 'full':
                previewHeight = '500px';
                break;
        }
        preview.css('height', previewHeight);
        
        // テキストカラーを適用
        preview.find('h1, p').css('color', textColor);
        
        // 背景をリセット
        preview.css({
            'background': '',
            'background-image': '',
            'background-size': '',
            'background-position': '',
            'background-attachment': ''
        });
        
        if (type === 'gradient') {
            const start = $('#gradientStart').val();
            const end = $('#gradientEnd').val();
            const direction = $('#gradientDirection').val();
            preview.css('background', `linear-gradient(${direction}, ${start} 0%, ${end} 100%)`);
        } else if (type === 'image') {
            // 画像プレビューは実際のアップロード後に表示
            preview.css('background', '#666');
            // 既存のヘルプテキストを削除してから追加
            preview.find('.image-upload-help').remove();
            preview.append('<div class="image-upload-help" style="position: absolute; top: 10px; left: 10px; background: rgba(0,0,0,0.5); color: white; padding: 5px 10px; border-radius: 4px; font-size: 12px;">画像アップロード後にプレビュー表示</div>');
        } else if (type.startsWith('pattern-')) {
            const patternColor = $('#patternColor').val();
            const bgColor = $('#patternBackground').val();
            const speed = $('#patternSpeed').val();
            preview.css('background', bgColor);
            
            // アニメーション速度の設定
            let animationDuration = '20s'; // normal
            if (speed === 'slow') {
                animationDuration = '30s';
            } else if (speed === 'fast') {
                animationDuration = '10s';
            }
            
            // パターンごとの簡易プレビュー
            let patternHtml = '';
            if (type === 'pattern-particles') {
                patternHtml = `<div class="pattern-preview-particles" style="position: absolute; width: 100%; height: 100%; overflow: hidden;">
                    <div class="particle" style="position: absolute; width: 10px; height: 10px; background: ${patternColor}; border-radius: 50%; top: 20%; left: 10%; opacity: 0.6; animation: float ${animationDuration} infinite ease-in-out;"></div>
                    <div class="particle" style="position: absolute; width: 8px; height: 8px; background: ${patternColor}; border-radius: 50%; top: 50%; left: 30%; opacity: 0.4; animation: float ${animationDuration} infinite ease-in-out 1s;"></div>
                    <div class="particle" style="position: absolute; width: 12px; height: 12px; background: ${patternColor}; border-radius: 50%; top: 70%; left: 60%; opacity: 0.7; animation: float ${animationDuration} infinite ease-in-out 2s;"></div>
                    <div class="particle" style="position: absolute; width: 6px; height: 6px; background: ${patternColor}; border-radius: 50%; top: 30%; left: 80%; opacity: 0.5; animation: float ${animationDuration} infinite ease-in-out 3s;"></div>
                </div>`;
            } else if (type === 'pattern-waves') {
                patternHtml = `<div class="pattern-preview-waves" style="position: absolute; bottom: 0; width: 100%; height: 50%; background: linear-gradient(0deg, ${patternColor}33 0%, transparent 100%); animation: wave ${animationDuration} infinite ease-in-out;"></div>`;
            } else if (type === 'pattern-geometric') {
                patternHtml = `<div class="pattern-preview-geometric" style="position: absolute; width: 100%; height: 100%; opacity: 0.1;">
                    <div style="position: absolute; width: 100px; height: 100px; border: 2px solid ${patternColor}; transform: rotate(45deg); top: 20%; left: 20%; animation: rotate ${animationDuration} infinite linear;"></div>
                    <div style="position: absolute; width: 80px; height: 80px; border: 2px solid ${patternColor}; transform: rotate(30deg); top: 50%; left: 60%; animation: rotate ${animationDuration} infinite linear reverse;"></div>
                </div>`;
            } else if (type === 'pattern-stars') {
                patternHtml = `<div class="pattern-preview-stars" style="position: absolute; width: 100%; height: 100%; overflow: hidden;">
                    <div style="position: absolute; width: 2px; height: 2px; background: ${patternColor}; top: 10%; left: 20%; animation: twinkle ${animationDuration} infinite;"></div>
                    <div style="position: absolute; width: 3px; height: 3px; background: ${patternColor}; top: 30%; left: 70%; animation: twinkle ${animationDuration} infinite 0.5s;"></div>
                    <div style="position: absolute; width: 2px; height: 2px; background: ${patternColor}; top: 60%; left: 40%; animation: twinkle ${animationDuration} infinite 1s;"></div>
                    <div style="position: absolute; width: 3px; height: 3px; background: ${patternColor}; top: 80%; left: 85%; animation: twinkle ${animationDuration} infinite 1.5s;"></div>
                </div>`;
            } else if (type === 'pattern-aurora') {
                patternHtml = `<div class="pattern-preview-aurora" style="position: absolute; width: 100%; height: 100%; overflow: hidden;">
                    <div style="position: absolute; width: 200%; height: 100%; background: linear-gradient(90deg, transparent 0%, ${patternColor}44 25%, ${patternColor}66 50%, ${patternColor}44 75%, transparent 100%); animation: aurora ${animationDuration} infinite ease-in-out;"></div>
                    <div style="position: absolute; width: 200%; height: 100%; background: linear-gradient(120deg, transparent 0%, ${patternColor}33 30%, transparent 60%); animation: aurora ${animationDuration} infinite ease-in-out reverse;"></div>
                </div>`;
            } else if (type === 'pattern-matrix') {
                patternHtml = `<div class="pattern-preview-matrix" style="position: absolute; width: 100%; height: 100%; overflow: hidden; font-family: monospace;">
                    <div style="position: absolute; color: ${patternColor}; font-size: 12px; top: -20px; left: 10%; animation: matrix-fall ${animationDuration} infinite linear;">01</div>
                    <div style="position: absolute; color: ${patternColor}; font-size: 12px; top: -20px; left: 30%; animation: matrix-fall ${animationDuration} infinite linear 0.5s;">10</div>
                    <div style="position: absolute; color: ${patternColor}; font-size: 12px; top: -20px; left: 50%; animation: matrix-fall ${animationDuration} infinite linear 1s;">11</div>
                    <div style="position: absolute; color: ${patternColor}; font-size: 12px; top: -20px; left: 70%; animation: matrix-fall ${animationDuration} infinite linear 1.5s;">00</div>
                    <div style="position: absolute; color: ${patternColor}; font-size: 12px; top: -20px; left: 90%; animation: matrix-fall ${animationDuration} infinite linear 2s;">01</div>
                </div>`;
            } else if (type === 'pattern-bubbles') {
                patternHtml = `<div class="pattern-preview-bubbles" style="position: absolute; width: 100%; height: 100%; overflow: hidden;">
                    <div style="position: absolute; width: 40px; height: 40px; background: ${patternColor}22; border: 2px solid ${patternColor}44; border-radius: 50%; bottom: -50px; left: 10%; animation: bubble-rise ${animationDuration} infinite ease-in-out;"></div>
                    <div style="position: absolute; width: 25px; height: 25px; background: ${patternColor}33; border: 2px solid ${patternColor}55; border-radius: 50%; bottom: -50px; left: 40%; animation: bubble-rise ${animationDuration} infinite ease-in-out 1s;"></div>
                    <div style="position: absolute; width: 35px; height: 35px; background: ${patternColor}22; border: 2px solid ${patternColor}44; border-radius: 50%; bottom: -50px; left: 70%; animation: bubble-rise ${animationDuration} infinite ease-in-out 2s;"></div>
                </div>`;
            } else if (type === 'pattern-hexagon') {
                patternHtml = `<div class="pattern-preview-hexagon" style="position: absolute; width: 100%; height: 100%; overflow: hidden;">
                    <div style="position: absolute; width: 60px; height: 60px; background: ${patternColor}; clip-path: polygon(50% 0%, 100% 25%, 100% 75%, 50% 100%, 0% 75%, 0% 25%); top: 20%; left: 20%; opacity: 0.2; animation: pulse ${animationDuration} infinite ease-in-out;"></div>
                    <div style="position: absolute; width: 60px; height: 60px; background: ${patternColor}; clip-path: polygon(50% 0%, 100% 25%, 100% 75%, 50% 100%, 0% 75%, 0% 25%); top: 50%; left: 60%; opacity: 0.3; animation: pulse ${animationDuration} infinite ease-in-out 1s;"></div>
                </div>`;
            } else if (type === 'pattern-circuit') {
                patternHtml = `<div class="pattern-preview-circuit" style="position: absolute; width: 100%; height: 100%; overflow: hidden;">
                    <div style="position: absolute; width: 100%; height: 2px; background: ${patternColor}44; top: 30%; animation: circuit-flow ${animationDuration} infinite linear;"></div>
                    <div style="position: absolute; width: 2px; height: 100%; background: ${patternColor}44; left: 40%; animation: circuit-flow-vertical ${animationDuration} infinite linear;"></div>
                    <div style="position: absolute; width: 10px; height: 10px; background: ${patternColor}; border-radius: 50%; top: 30%; left: 40%; box-shadow: 0 0 10px ${patternColor};"></div>
                </div>`;
            } else if (type === 'pattern-galaxy') {
                patternHtml = `<div class="pattern-preview-galaxy" style="position: absolute; width: 100%; height: 100%; overflow: hidden;">
                    <div style="position: absolute; width: 100%; height: 100%; background: radial-gradient(ellipse at center, ${patternColor}44 0%, transparent 70%); animation: galaxy-rotate ${animationDuration} infinite linear;"></div>
                    <div style="position: absolute; width: 4px; height: 4px; background: ${patternColor}; border-radius: 50%; top: 40%; left: 30%; animation: orbit ${animationDuration} infinite linear;"></div>
                    <div style="position: absolute; width: 3px; height: 3px; background: ${patternColor}; border-radius: 50%; top: 60%; left: 70%; animation: orbit ${animationDuration} infinite linear reverse;"></div>
                </div>`;
            } else if (type === 'pattern-rain') {
                patternHtml = `<div class="pattern-preview-rain" style="position: absolute; width: 100%; height: 100%; overflow: hidden;">
                    <div style="position: absolute; width: 2px; height: 20px; background: linear-gradient(to bottom, transparent, ${patternColor}); top: -20px; left: 20%; animation: rain-fall ${animationDuration} infinite linear;"></div>
                    <div style="position: absolute; width: 2px; height: 15px; background: linear-gradient(to bottom, transparent, ${patternColor}); top: -20px; left: 40%; animation: rain-fall ${animationDuration} infinite linear 0.5s;"></div>
                    <div style="position: absolute; width: 2px; height: 25px; background: linear-gradient(to bottom, transparent, ${patternColor}); top: -20px; left: 60%; animation: rain-fall ${animationDuration} infinite linear 1s;"></div>
                    <div style="position: absolute; width: 2px; height: 18px; background: linear-gradient(to bottom, transparent, ${patternColor}); top: -20px; left: 80%; animation: rain-fall ${animationDuration} infinite linear 1.5s;"></div>
                </div>`;
            } else if (type === 'pattern-fire') {
                patternHtml = `<div class="pattern-preview-fire" style="position: absolute; width: 100%; height: 100%; overflow: hidden;">
                    <div style="position: absolute; bottom: 0; width: 100%; height: 60%; background: linear-gradient(to top, ${patternColor}66, ${patternColor}33, transparent); animation: fire-flicker ${animationDuration} infinite ease-in-out;"></div>
                    <div style="position: absolute; bottom: 0; left: 20%; width: 30%; height: 40%; background: radial-gradient(ellipse at bottom, ${patternColor}88, transparent); animation: fire-dance ${animationDuration} infinite ease-in-out;"></div>
                    <div style="position: absolute; bottom: 0; right: 20%; width: 30%; height: 40%; background: radial-gradient(ellipse at bottom, ${patternColor}88, transparent); animation: fire-dance ${animationDuration} infinite ease-in-out reverse;"></div>
                </div>`;
            } else if (type === 'pattern-snow') {
                patternHtml = `<div class="pattern-preview-snow" style="position: absolute; width: 100%; height: 100%; overflow: hidden;">
                    <div style="position: absolute; width: 8px; height: 8px; background: ${patternColor}; border-radius: 50%; top: -10px; left: 10%; animation: snow-fall ${animationDuration} infinite linear;"></div>
                    <div style="position: absolute; width: 6px; height: 6px; background: ${patternColor}; border-radius: 50%; top: -10px; left: 30%; animation: snow-fall ${animationDuration} infinite linear 1s;"></div>
                    <div style="position: absolute; width: 10px; height: 10px; background: ${patternColor}; border-radius: 50%; top: -10px; left: 50%; animation: snow-fall ${animationDuration} infinite linear 2s;"></div>
                    <div style="position: absolute; width: 7px; height: 7px; background: ${patternColor}; border-radius: 50%; top: -10px; left: 70%; animation: snow-fall ${animationDuration} infinite linear 0.5s;"></div>
                    <div style="position: absolute; width: 9px; height: 9px; background: ${patternColor}; border-radius: 50%; top: -10px; left: 90%; animation: snow-fall ${animationDuration} infinite linear 1.5s;"></div>
                </div>`;
            } else if (type === 'pattern-neural') {
                patternHtml = `<div class="pattern-preview-neural" style="position: absolute; width: 100%; height: 100%; overflow: hidden;">
                    <div style="position: absolute; width: 15px; height: 15px; background: ${patternColor}; border-radius: 50%; top: 30%; left: 20%; box-shadow: 0 0 20px ${patternColor}; animation: pulse ${animationDuration} infinite;"></div>
                    <div style="position: absolute; width: 15px; height: 15px; background: ${patternColor}; border-radius: 50%; top: 50%; left: 70%; box-shadow: 0 0 20px ${patternColor}; animation: pulse ${animationDuration} infinite 0.5s;"></div>
                    <div style="position: absolute; width: 2px; height: 50px; background: ${patternColor}44; top: 30%; left: 20%; transform-origin: top; transform: rotate(30deg);"></div>
                    <div style="position: absolute; width: 2px; height: 80px; background: ${patternColor}44; top: 30%; left: 20%; transform-origin: top; transform: rotate(-20deg);"></div>
                </div>`;
            } else if (type === 'pattern-mosaic') {
                patternHtml = `<div class="pattern-preview-mosaic" style="position: absolute; width: 100%; height: 100%; overflow: hidden;">
                    <div style="position: absolute; width: 50px; height: 50px; background: ${patternColor}22; top: 20%; left: 10%; animation: mosaic-fade ${animationDuration} infinite;"></div>
                    <div style="position: absolute; width: 50px; height: 50px; background: ${patternColor}33; top: 20%; left: 30%; animation: mosaic-fade ${animationDuration} infinite 0.5s;"></div>
                    <div style="position: absolute; width: 50px; height: 50px; background: ${patternColor}44; top: 40%; left: 50%; animation: mosaic-fade ${animationDuration} infinite 1s;"></div>
                    <div style="position: absolute; width: 50px; height: 50px; background: ${patternColor}33; top: 60%; left: 70%; animation: mosaic-fade ${animationDuration} infinite 1.5s;"></div>
                </div>`;
            } else if (type === 'pattern-sparkle') {
                patternHtml = `<div class="pattern-preview-sparkle" style="position: absolute; width: 100%; height: 100%; overflow: hidden;">
                    <div style="position: absolute; width: 4px; height: 4px; background: white; box-shadow: 0 0 10px white, 0 0 20px ${patternColor}; top: 20%; left: 15%; animation: sparkle ${animationDuration} infinite;"></div>
                    <div style="position: absolute; width: 3px; height: 3px; background: white; box-shadow: 0 0 8px white, 0 0 15px ${patternColor}; top: 40%; left: 30%; animation: sparkle ${animationDuration} infinite 0.3s;"></div>
                    <div style="position: absolute; width: 5px; height: 5px; background: white; box-shadow: 0 0 12px white, 0 0 25px ${patternColor}; top: 60%; left: 45%; animation: sparkle ${animationDuration} infinite 0.6s;"></div>
                    <div style="position: absolute; width: 3px; height: 3px; background: white; box-shadow: 0 0 8px white, 0 0 15px ${patternColor}; top: 35%; left: 65%; animation: sparkle ${animationDuration} infinite 0.9s;"></div>
                    <div style="position: absolute; width: 4px; height: 4px; background: white; box-shadow: 0 0 10px white, 0 0 20px ${patternColor}; top: 70%; left: 80%; animation: sparkle ${animationDuration} infinite 1.2s;"></div>
                    <div style="position: absolute; width: 2px; height: 2px; background: white; box-shadow: 0 0 6px white, 0 0 12px ${patternColor}; top: 15%; left: 50%; animation: sparkle ${animationDuration} infinite 0.15s;"></div>
                    <div style="position: absolute; width: 3px; height: 3px; background: white; box-shadow: 0 0 8px white, 0 0 15px ${patternColor}; top: 85%; left: 25%; animation: sparkle ${animationDuration} infinite 0.45s;"></div>
                    <div style="position: absolute; width: 4px; height: 4px; background: white; box-shadow: 0 0 10px white, 0 0 20px ${patternColor}; top: 50%; left: 90%; animation: sparkle ${animationDuration} infinite 0.75s;"></div>
                </div>`;
            } else if (type === 'pattern-crystal') {
                patternHtml = `<div class="pattern-preview-crystal" style="position: absolute; width: 100%; height: 100%; overflow: hidden;">
                    <div style="position: absolute; width: 20px; height: 20px; background: linear-gradient(45deg, transparent 30%, ${patternColor}44 50%, transparent 70%); transform: rotate(45deg); top: 25%; left: 20%; animation: crystal-rotate ${animationDuration} infinite linear;"></div>
                    <div style="position: absolute; width: 15px; height: 15px; background: linear-gradient(45deg, transparent 30%, ${patternColor}66 50%, transparent 70%); transform: rotate(45deg); top: 50%; left: 40%; animation: crystal-rotate ${animationDuration} infinite linear reverse;"></div>
                    <div style="position: absolute; width: 25px; height: 25px; background: linear-gradient(45deg, transparent 30%, ${patternColor}33 50%, transparent 70%); transform: rotate(45deg); top: 70%; left: 70%; animation: crystal-rotate ${animationDuration} infinite linear 0.5s;"></div>
                    <div style="position: absolute; width: 18px; height: 18px; background: linear-gradient(45deg, transparent 30%, ${patternColor}55 50%, transparent 70%); transform: rotate(45deg); top: 35%; left: 60%; animation: crystal-rotate ${animationDuration} infinite linear 1s;"></div>
                </div>`;
            } else if (type === 'pattern-sakura') {
                patternHtml = `<div class="pattern-preview-sakura" style="position: absolute; width: 100%; height: 100%; overflow: hidden;">
                    <div style="position: absolute; width: 15px; height: 15px; background: radial-gradient(circle, #ffb7c5 30%, transparent 70%); border-radius: 0 100% 0 100%; top: -20px; left: 10%; animation: sakura-fall ${animationDuration} infinite linear; transform: rotate(45deg);"></div>
                    <div style="position: absolute; width: 12px; height: 12px; background: radial-gradient(circle, #ffb7c5 30%, transparent 70%); border-radius: 0 100% 0 100%; top: -20px; left: 30%; animation: sakura-fall ${animationDuration} infinite linear 0.5s; transform: rotate(30deg);"></div>
                    <div style="position: absolute; width: 18px; height: 18px; background: radial-gradient(circle, #ffb7c5 30%, transparent 70%); border-radius: 0 100% 0 100%; top: -20px; left: 50%; animation: sakura-fall ${animationDuration} infinite linear 1s; transform: rotate(60deg);"></div>
                    <div style="position: absolute; width: 14px; height: 14px; background: radial-gradient(circle, #ffb7c5 30%, transparent 70%); border-radius: 0 100% 0 100%; top: -20px; left: 70%; animation: sakura-fall ${animationDuration} infinite linear 1.5s; transform: rotate(15deg);"></div>
                    <div style="position: absolute; width: 16px; height: 16px; background: radial-gradient(circle, #ffb7c5 30%, transparent 70%); border-radius: 0 100% 0 100%; top: -20px; left: 85%; animation: sakura-fall ${animationDuration} infinite linear 2s; transform: rotate(75deg);"></div>
                </div>`;
            } else if (type === 'pattern-firefly') {
                patternHtml = `<div class="pattern-preview-firefly" style="position: absolute; width: 100%; height: 100%; overflow: hidden;">
                    <div style="position: absolute; width: 6px; height: 6px; background: radial-gradient(circle, #ffff88 0%, #ffff00 50%, transparent 100%); border-radius: 50%; top: 30%; left: 20%; animation: firefly-move ${animationDuration} infinite; box-shadow: 0 0 10px #ffff88;"></div>
                    <div style="position: absolute; width: 5px; height: 5px; background: radial-gradient(circle, #ffff88 0%, #ffff00 50%, transparent 100%); border-radius: 50%; top: 60%; left: 40%; animation: firefly-move ${animationDuration} infinite 1s; box-shadow: 0 0 8px #ffff88;"></div>
                    <div style="position: absolute; width: 7px; height: 7px; background: radial-gradient(circle, #ffff88 0%, #ffff00 50%, transparent 100%); border-radius: 50%; top: 45%; left: 70%; animation: firefly-move ${animationDuration} infinite 2s; box-shadow: 0 0 12px #ffff88;"></div>
                    <div style="position: absolute; width: 4px; height: 4px; background: radial-gradient(circle, #ffff88 0%, #ffff00 50%, transparent 100%); border-radius: 50%; top: 75%; left: 85%; animation: firefly-move ${animationDuration} infinite 1.5s; box-shadow: 0 0 6px #ffff88;"></div>
                </div>`;
            } else if (type === 'pattern-rainbow') {
                patternHtml = `<div class="pattern-preview-rainbow" style="position: absolute; width: 100%; height: 100%; overflow: hidden;">
                    <div style="position: absolute; width: 100%; height: 20%; background: linear-gradient(90deg, rgba(255,0,0,0.3), rgba(255,127,0,0.3), rgba(255,255,0,0.3), rgba(0,255,0,0.3), rgba(0,0,255,0.3), rgba(75,0,130,0.3), rgba(148,0,211,0.3)); top: 20%; animation: rainbow-wave ${animationDuration} infinite linear;"></div>
                    <div style="position: absolute; width: 100%; height: 15%; background: linear-gradient(90deg, rgba(255,0,0,0.2), rgba(255,127,0,0.2), rgba(255,255,0,0.2), rgba(0,255,0,0.2), rgba(0,0,255,0.2), rgba(75,0,130,0.2), rgba(148,0,211,0.2)); top: 45%; animation: rainbow-wave ${animationDuration} infinite linear 0.5s;"></div>
                    <div style="position: absolute; width: 100%; height: 25%; background: linear-gradient(90deg, rgba(255,0,0,0.25), rgba(255,127,0,0.25), rgba(255,255,0,0.25), rgba(0,255,0,0.25), rgba(0,0,255,0.25), rgba(75,0,130,0.25), rgba(148,0,211,0.25)); top: 70%; animation: rainbow-wave ${animationDuration} infinite linear 1s;"></div>
                </div>`;
            } else if (type === 'pattern-diamond') {
                patternHtml = `<div class="pattern-preview-diamond" style="position: absolute; width: 100%; height: 100%; overflow: hidden;">
                    <div style="position: absolute; width: 15px; height: 15px; background: linear-gradient(45deg, ${patternColor}88 25%, transparent 25%, transparent 75%, ${patternColor}88 75%); transform: rotate(45deg); top: 20%; left: 25%; animation: diamond-shine ${animationDuration} infinite;"></div>
                    <div style="position: absolute; width: 20px; height: 20px; background: linear-gradient(45deg, ${patternColor}66 25%, transparent 25%, transparent 75%, ${patternColor}66 75%); transform: rotate(45deg); top: 40%; left: 50%; animation: diamond-shine ${animationDuration} infinite 0.5s;"></div>
                    <div style="position: absolute; width: 12px; height: 12px; background: linear-gradient(45deg, ${patternColor}aa 25%, transparent 25%, transparent 75%, ${patternColor}aa 75%); transform: rotate(45deg); top: 65%; left: 30%; animation: diamond-shine ${animationDuration} infinite 1s;"></div>
                    <div style="position: absolute; width: 18px; height: 18px; background: linear-gradient(45deg, ${patternColor}77 25%, transparent 25%, transparent 75%, ${patternColor}77 75%); transform: rotate(45deg); top: 35%; left: 70%; animation: diamond-shine ${animationDuration} infinite 1.5s;"></div>
                </div>`;
            } else if (type === 'pattern-glitter') {
                patternHtml = `<div class="pattern-preview-glitter" style="position: absolute; width: 100%; height: 100%; overflow: hidden; background: radial-gradient(ellipse at center, transparent 0%, rgba(255,255,255,0.1) 100%);">
                    <div style="position: absolute; width: 2px; height: 2px; background: white; top: 15%; left: 20%; animation: glitter-twinkle ${animationDuration} infinite;"></div>
                    <div style="position: absolute; width: 3px; height: 3px; background: white; top: 25%; left: 35%; animation: glitter-twinkle ${animationDuration} infinite 0.2s;"></div>
                    <div style="position: absolute; width: 2px; height: 2px; background: white; top: 40%; left: 15%; animation: glitter-twinkle ${animationDuration} infinite 0.4s;"></div>
                    <div style="position: absolute; width: 1px; height: 1px; background: white; top: 30%; left: 60%; animation: glitter-twinkle ${animationDuration} infinite 0.6s;"></div>
                    <div style="position: absolute; width: 3px; height: 3px; background: white; top: 55%; left: 45%; animation: glitter-twinkle ${animationDuration} infinite 0.8s;"></div>
                    <div style="position: absolute; width: 2px; height: 2px; background: white; top: 70%; left: 25%; animation: glitter-twinkle ${animationDuration} infinite 1s;"></div>
                    <div style="position: absolute; width: 1px; height: 1px; background: white; top: 60%; left: 75%; animation: glitter-twinkle ${animationDuration} infinite 1.2s;"></div>
                    <div style="position: absolute; width: 2px; height: 2px; background: white; top: 80%; left: 55%; animation: glitter-twinkle ${animationDuration} infinite 1.4s;"></div>
                    <div style="position: absolute; width: 3px; height: 3px; background: white; top: 45%; left: 85%; animation: glitter-twinkle ${animationDuration} infinite 1.6s;"></div>
                </div>`;
            } else if (type === 'pattern-stardust') {
                patternHtml = `<div class="pattern-preview-stardust" style="position: absolute; width: 100%; height: 100%; overflow: hidden;">
                    <div style="position: absolute; width: 100%; height: 100%; background: radial-gradient(circle at 20% 30%, ${patternColor}22 0%, transparent 50%), radial-gradient(circle at 70% 60%, ${patternColor}33 0%, transparent 40%), radial-gradient(circle at 40% 80%, ${patternColor}22 0%, transparent 45%); animation: stardust-drift ${animationDuration} infinite linear;"></div>
                    <div style="position: absolute; width: 3px; height: 3px; background: white; border-radius: 50%; top: 20%; left: 30%; box-shadow: 0 0 6px white, 0 0 12px ${patternColor}; animation: star-pulse ${animationDuration} infinite;"></div>
                    <div style="position: absolute; width: 2px; height: 2px; background: white; border-radius: 50%; top: 50%; left: 60%; box-shadow: 0 0 4px white, 0 0 8px ${patternColor}; animation: star-pulse ${animationDuration} infinite 0.5s;"></div>
                    <div style="position: absolute; width: 4px; height: 4px; background: white; border-radius: 50%; top: 70%; left: 40%; box-shadow: 0 0 8px white, 0 0 16px ${patternColor}; animation: star-pulse ${animationDuration} infinite 1s;"></div>
                </div>`;
            }
            
            if (patternHtml) {
                // 既存のパターンプレビューを削除
                preview.find('[class^="pattern-preview-"]').remove();
                // パターンをヘッダーコンテンツの前に挿入（z-indexで制御）
                preview.find('.header-preview-content').before(patternHtml);
                // パターンプレビューのz-indexを設定
                preview.find('[class^="pattern-preview-"]').css('z-index', '1');
            }
        }
    }
    
    // ヘッダー設定変更時のプレビュー更新
    $('#gradientStart, #gradientEnd, #gradientDirection, #patternColor, #patternBackground, #headerTextColor, #headerHeight, #patternSpeed').change(updateHeaderPreview);
    
    // 画像アップロード時のプレビュー
    $('#headerImage').change(function() {
        const file = this.files[0];
        if (file) {
            const reader = new FileReader();
            reader.onload = function(e) {
                $('#headerPreview').css({
                    'background-image': `url(${e.target.result})`,
                    'background-size': 'cover',
                    'background-position': 'center'
                });
            };
            reader.readAsDataURL(file);
        }
    });
    
    // 見出しスタイル変更時のプレビュー更新
    $('#h1Style, #h1Color, #h2Style, #h2Color, #h3Style, #h3Color, #h4Style, #h4Color').change(updateHeadingPreview);
    
    function updateHeadingPreview() {
        // H1のスタイル適用
        const h1Style = $('#h1Style').val();
        const h1Color = $('#h1Color').val();
        applyHeadingStyle('.preview-h1', h1Style, h1Color);
        
        // H2のスタイル適用
        const h2Style = $('#h2Style').val();
        const h2Color = $('#h2Color').val();
        applyHeadingStyle('.preview-h2', h2Style, h2Color);
        
        // H3のスタイル適用
        const h3Style = $('#h3Style').val();
        const h3Color = $('#h3Color').val();
        applyHeadingStyle('.preview-h3', h3Style, h3Color);
        
        // H4のスタイル適用
        const h4Style = $('#h4Style').val();
        const h4Color = $('#h4Color').val();
        applyHeadingStyle('.preview-h4', h4Style, h4Color);
    }
    
    function applyHeadingStyle(selector, style, color) {
        const element = $(selector);
        
        // スタイルをリセット（重要：with-checkmarkクラスも削除）
        element.removeClass('with-checkmark');
        element.css({
            'background': 'none',
            'border': 'none',
            'padding': '0',
            'position': 'static', // relativeからstaticに変更
            'padding-left': '0'
        });
        
        switch(style) {
            case 'filled':
                element.css({
                    'background': color,
                    'color': '#fff',
                    'padding': '10px 15px',
                    'border-radius': '4px'
                });
                break;
            case 'underline':
                element.css({
                    'border-bottom': `3px solid ${color}`,
                    'padding-bottom': '5px',
                    'color': color
                });
                break;
            case 'checkmark':
                element.css({
                    'padding-left': '30px',
                    'position': 'relative',
                    'color': color
                });
                // チェックマークを擬似要素で追加（CSSクラスを使用）
                element.addClass('with-checkmark');
                break;
            default:
                element.css('color', color);
        }
    }
    
    // カラー変更時のプレビュー更新
    $('#primaryColor, #secondaryColor, #accentColor, #textColor, #linkColor, #linkHoverColor').change(updateColorPreview);
    
    function updateColorPreview() {
        $('#primaryPreview').css('background-color', $('#primaryColor').val());
        $('#secondaryPreview').css('background-color', $('#secondaryColor').val());
        $('#accentPreview').css('background-color', $('#accentColor').val());
        $('#textPreview').css('color', $('#textColor').val());
        $('#linkPreview').css('color', $('#linkColor').val());
        
        // ホバー効果のためのスタイル追加
        $('#linkPreview').hover(
            function() { $(this).css('color', $('#linkHoverColor').val()); },
            function() { $(this).css('color', $('#linkColor').val()); }
        );
    }
    
    // デザイン設定の保存
    $('#saveDesignSettings').click(function() {
        // textareaの内容を直接取得（TinyMCEは使用しない）
        
        // 基本情報を収集
        const basicInfo = {
            display_name: $('#siteDisplayName').val(),
            description: $('#siteDescription').val(),
            hero_catchphrase: $('#heroCatchphrase').val(),
            seo_keywords: $('#siteKeywords').val(),
            meta_description: $('#siteMetaDescription').val(),
            ogp_title: $('#ogpTitle').val(),
            ogp_description: $('#ogpDescription').val()
        };
        
        // 設定オブジェクトの作成
        const designSettings = {
            template: $('#columnLayout').val(), // レイアウト（カラム数）設定を追加
            free_spaces: {
                top_upper: $('#topUpperFreeSpace').val(),
                top_lower: $('#topLowerFreeSpace').val(),
                article_upper: $('#articleUpperFreeSpace').val(),
                article_lower: $('#articleLowerFreeSpace').val()
            },
            header_design: {
                type: $('#headerType').val(),
                height: $('#headerHeight').val(),
                text_color: $('#headerTextColor').val(),
                gradient: {
                    start: $('#gradientStart').val(),
                    end: $('#gradientEnd').val(),
                    direction: $('#gradientDirection').val()
                },
                image: {
                    display: $('#headerImageDisplay').val(),
                    overlay: $('#headerImageOverlay').val()
                },
                pattern: {
                    color: $('#patternColor').val(),
                    background: $('#patternBackground').val(),
                    speed: $('#patternSpeed').val()
                }
            },
            heading_styles: {
                h1_style: $('#h1Style').val(),
                h1_color: $('#h1Color').val(),
                h2_style: $('#h2Style').val(),
                h2_color: $('#h2Color').val(),
                h3_style: $('#h3Style').val(),
                h3_color: $('#h3Color').val(),
                h4_style: $('#h4Style').val(),
                h4_color: $('#h4Color').val()
            },
            list_styles: {
                ul_style: $('#ulStyle').val(),
                ul_color: $('#ulColor').val(),
                ul_custom_char: $('#ulCustomChar').val(),
                ol_style: $('#olStyle').val(),
                ol_color: $('#olColor').val(),
                animation_enabled: $('#listAnimation').is(':checked'),
                animation_type: $('#listAnimationType').val()
            },
            backgrounds: {
                page_bg_color: $('#pageBackgroundColor').val(),
                page_bg_image: '', // 画像アップロード処理は別途実装
                page_bg_repeat: $('#pageBackgroundRepeat').val(),
                page_bg_position: $('#pageBackgroundPosition').val(),
                article_bg_color: $('#articleBackgroundColor').val(),
                article_bg_image: '', // 画像アップロード処理は別途実装
                article_bg_repeat: $('#articleBackgroundRepeat').val(),
                article_bg_position: $('#articleBackgroundPosition').val(),
                animation_style: $('#backgroundType').val() === 'normal' ? 'none' : ($('#animationStyle').val() || 'none'),
                anim_color1: $('#animColor1').val() || '#667eea',
                anim_color2: $('#animColor2').val() || '#764ba2',
                anim_speed: $('#animSpeed').val() || '5'
            },
            theme_colors: {
                primary: $('#primaryColor').val(),
                secondary: $('#secondaryColor').val(),
                accent: $('#accentColor').val(),
                text: $('#textColor').val(),
                link: $('#linkColor').val(),
                link_hover: $('#linkHoverColor').val()
            },
            verification_codes: {
                google_search_console: $('#googleSearchConsole').val(),
                google_analytics: $('#googleAnalytics').val(),
                facebook_pixel: $('#facebookPixel').val(),
                custom_head_code: $('#customHeadCode').val()
            }
        };
        
        // 画像アップロード処理
        const formData = new FormData();
        formData.append('config_id', window.currentDesignConfigId);
        formData.append('design_settings', JSON.stringify(designSettings));
        formData.append('basic_info', JSON.stringify(basicInfo));
        formData.append('articles_per_page', $('#articlesPerPage').val());
        
        // ヘッダー画像がある場合
        const headerImage = $('#headerImage')[0].files[0];
        if (headerImage) {
            formData.append('header_image', headerImage);
        }
        
        // 背景画像がある場合
        const pageImage = $('#pageBackgroundImage')[0].files[0];
        if (pageImage) {
            formData.append('page_bg_image', pageImage);
        }
        
        const articleImage = $('#articleBackgroundImage')[0].files[0];
        if (articleImage) {
            formData.append('article_bg_image', articleImage);
        }
        
        // APIで設定を保存
        $.ajax({
            url: '../api/update-site-design.php',
            method: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                if (response.success) {
                    alert('デザイン設定を保存しました');
                    
                    // ローカルデータを更新
                    siteConfigs[window.currentDesignConfigId].design_settings = response.design_settings;
                    
                    // 基本情報もローカルデータに反映
                    if (basicInfo.display_name !== undefined) {
                        siteConfigs[window.currentDesignConfigId].display_name = basicInfo.display_name;
                    }
                    if (basicInfo.description !== undefined) {
                        siteConfigs[window.currentDesignConfigId].description = basicInfo.description;
                    }
                    if (basicInfo.hero_catchphrase !== undefined) {
                        siteConfigs[window.currentDesignConfigId].hero_catchphrase = basicInfo.hero_catchphrase;
                    }
                    if (basicInfo.seo_keywords !== undefined) {
                        siteConfigs[window.currentDesignConfigId].seo_keywords = basicInfo.seo_keywords;
                    }
                    if (basicInfo.meta_description !== undefined) {
                        siteConfigs[window.currentDesignConfigId].meta_description = basicInfo.meta_description;
                    }
                    if (basicInfo.ogp_title !== undefined) {
                        siteConfigs[window.currentDesignConfigId].ogp_title = basicInfo.ogp_title;
                    }
                    if (basicInfo.ogp_description !== undefined) {
                        siteConfigs[window.currentDesignConfigId].ogp_description = basicInfo.ogp_description;
                    }
                    
                    // モーダルを閉じる
                    bootstrap.Modal.getInstance(document.getElementById('designSettingsModal')).hide();
                } else {
                    alert('エラー: ' + response.error);
                }
            },
            error: function() {
                alert('通信エラーが発生しました');
            }
        });
    });
    
    // リストスタイル関連の処理
    $('#ulStyle').change(function() {
        const style = $(this).val();
        if (style === 'custom') {
            $('#ulCustomChar').show();
        } else {
            $('#ulCustomChar').hide();
        }
        updateListPreview();
    });
    
    $('#ulColor, #olStyle, #olColor').change(updateListPreview);
    
    $('#listAnimation').change(function() {
        if ($(this).is(':checked')) {
            $('#listAnimationType').show();
        } else {
            $('#listAnimationType').hide();
        }
        updateListPreview();
    });
    
    function updateListPreview() {
        const ulStyle = $('#ulStyle').val();
        const ulColor = $('#ulColor').val();
        const olStyle = $('#olStyle').val();
        const olColor = $('#olColor').val();
        
        // ULスタイルの適用
        let ulChar = '';
        switch(ulStyle) {
            case 'check': ulChar = '✓'; break;
            case 'arrow': ulChar = '➜'; break;
            case 'star': ulChar = '★'; break;
            case 'heart': ulChar = '♥'; break;
            case 'circle': ulChar = '●'; break;
            case 'square': ulChar = '■'; break;
            case 'diamond': ulChar = '◆'; break;
            case 'hand': ulChar = '☞'; break;
            case 'custom': ulChar = $('#ulCustomChar').val() || '•'; break;
            default: ulChar = '•';
        }
        
        // ULのプレビュー更新
        $('.preview-ul').css({
            'list-style': 'none',
            'padding-left': '30px'
        });
        
        $('.preview-ul li').each(function() {
            $(this).css({
                'position': 'relative',
                'padding-left': '25px'
            });
            
            // 既存の疑似要素を削除して新しく追加
            $(this).find('.list-marker').remove();
            $(this).prepend(`<span class="list-marker" style="position: absolute; left: 0; color: ${ulColor};">${ulChar}</span>`);
        });
        
        // OLスタイルの適用
        let olType = 'decimal';
        switch(olStyle) {
            case 'circle': olType = 'decimal'; break;
            case 'square': olType = 'decimal'; break;
            case 'roman': olType = 'upper-roman'; break;
            case 'alpha': olType = 'upper-alpha'; break;
            default: olType = 'decimal';
        }
        
        $('.preview-ol').css({
            'list-style-type': olType,
            'color': olColor
        });
        
        // アニメーション効果のプレビュー
        if ($('#listAnimation').is(':checked')) {
            const animType = $('#listAnimationType').val();
            $('.preview-ul li, .preview-ol li').addClass('list-' + animType);
        } else {
            $('.preview-ul li, .preview-ol li').removeClass('list-fade list-slide list-bounce list-rotate');
        }
    }
    
    // 背景タイプ切り替え
    $('#backgroundType').change(function() {
        if ($(this).val() === 'animated') {
            $('#normalBackgroundSettings').hide();
            $('#animatedBackgroundSettings').show();
            updateBackgroundPreview();
        } else {
            // 通常背景に切り替えた時は、アニメーションスタイルを「なし」にリセット
            $('#normalBackgroundSettings').show();
            $('#animatedBackgroundSettings').hide();
            $('#animationStyle').val('none');
            updateBackgroundPreview();
        }
    });
    
    // 背景色変更時のプレビュー更新
    $('#pageBackgroundColor, #animationStyle, #animColor1, #animColor2, #animSpeed').change(updateBackgroundPreview);
    
    function updateBackgroundPreview() {
        const bgType = $('#backgroundType').val();
        const preview = $('#backgroundPreview');
        const animatedBg = $('#animatedBgPreview');
        
        if (bgType === 'animated') {
            const style = $('#animationStyle').val();
            const color1 = $('#animColor1').val();
            const color2 = $('#animColor2').val();
            const speed = $('#animSpeed').val();
            
            animatedBg.show();
            
            // アニメーションスタイルに応じたプレビュー
            switch(style) {
                case 'gradient-wave':
                    animatedBg.css({
                        'background': `linear-gradient(45deg, ${color1}, ${color2})`,
                        'background-size': '200% 200%',
                        'animation': `gradientWave ${11 - speed}s ease infinite`
                    });
                    break;
                    
                case 'aurora-lights':
                    animatedBg.css({
                        'background': `radial-gradient(ellipse at bottom, ${color1} 0%, transparent 70%), radial-gradient(ellipse at top, ${color2} 0%, transparent 70%)`,
                        'animation': `aurora ${11 - speed}s ease infinite`
                    });
                    break;
                    
                case 'star-field':
                    animatedBg.css({
                        'background': `radial-gradient(2px 2px at 20% 30%, white, transparent), radial-gradient(2px 2px at 60% 70%, white, transparent), radial-gradient(1px 1px at 50% 50%, white, transparent)`,
                        'background-color': color1,
                        'background-size': '50px 50px',
                        'animation': `starMove ${11 - speed}s linear infinite`
                    });
                    break;
                    
                case 'cyber-rain':
                    animatedBg.css({
                        'background': `linear-gradient(180deg, ${color1} 0%, ${color2} 50%, ${color1} 100%)`,
                        'background-size': '100% 200%',
                        'animation': `cyberRain ${11 - speed}s linear infinite`
                    });
                    break;
                    
                case 'holographic-wave':
                    animatedBg.css({
                        'background': `linear-gradient(45deg, ${color1}, ${color2}, ${color1}, ${color2})`,
                        'background-size': '400% 400%',
                        'animation': `holographicWave ${11 - speed}s ease infinite`
                    });
                    break;
                    
                case 'plasma-storm':
                    animatedBg.css({
                        'background': `radial-gradient(circle at 20% 80%, ${color1} 0%, transparent 50%), radial-gradient(circle at 80% 20%, ${color2} 0%, transparent 50%), radial-gradient(circle at 40% 40%, ${color1} 0%, transparent 50%)`,
                        'animation': `plasmaStorm ${11 - speed}s ease-in-out infinite`
                    });
                    break;
                    
                case 'quantum-field':
                    animatedBg.css({
                        'background': `conic-gradient(from 0deg at 50% 50%, ${color1}, ${color2}, ${color1})`,
                        'animation': `quantumField ${11 - speed}s linear infinite`
                    });
                    break;
                    
                case 'nebula-cloud':
                    animatedBg.css({
                        'background': `radial-gradient(ellipse at top, ${color1} 0%, transparent 70%), radial-gradient(ellipse at bottom, ${color2} 0%, transparent 70%), linear-gradient(180deg, transparent 0%, ${color1}33 100%)`,
                        'animation': `nebulaCloud ${11 - speed}s ease-in-out infinite`
                    });
                    break;
                    
                default:
                    animatedBg.css({
                        'background': `linear-gradient(135deg, ${color1}, ${color2})`,
                        'animation': `pulse ${11 - speed}s ease infinite`
                    });
            }
        } else {
            animatedBg.hide();
            const bgColor = $('#pageBackgroundColor').val();
            preview.css('background-color', bgColor);
        }
    }
});
</script>

<style>
/* 背景アニメーションプレビュー用 */
@keyframes gradientWave {
    0% { background-position: 0% 50%; }
    50% { background-position: 100% 50%; }
    100% { background-position: 0% 50%; }
}

@keyframes aurora {
    0%, 100% { opacity: 0.8; transform: translateY(0); }
    50% { opacity: 1; transform: translateY(-10px); }
}

@keyframes starMove {
    from { background-position: 0 0; }
    to { background-position: 50px 50px; }
}

@keyframes pulse {
    0%, 100% { opacity: 1; }
    50% { opacity: 0.8; }
}

/* プレビューコンテナ */
#backgroundPreview {
    transition: background-color 0.3s ease;
}

#animatedBgPreview {
    transition: opacity 0.3s ease;
}
</style>

<style>
.card {
    transition: transform 0.2s;
}
.card:hover {
    transform: translateY(-5px);
    box-shadow: 0 5px 15px rgba(0,0,0,0.1);
}
.update-status {
    min-height: 20px;
}

/* チェックマーク付き見出しのスタイル（管理画面のプレビュー用） */
#headingPreview .with-checkmark::before {
    content: '✓';
    position: absolute;
    left: 0;
    font-weight: bold;
    font-size: 1.2em;
}

/* デザイン設定モーダルのスタイル */
#designSettingsModal .modal-dialog {
    max-height: 90vh;
}

#designSettingsModal .modal-content {
    height: 85vh;
    display: flex;
    flex-direction: column;
}

#designSettingsModal .modal-header {
    flex-shrink: 0;
}

#designSettingsModal .modal-body {
    flex: 1;
    overflow-y: auto;
    padding: 1rem 1.5rem;
}

#designSettingsModal .modal-footer {
    flex-shrink: 0;
}

#designSettingsModal .nav-tabs {
    border-bottom: 2px solid #dee2e6;
    position: sticky;
    top: 0;
    background: white;
    z-index: 10;
    padding-top: 0.5rem;
    margin-top: -0.5rem;
}

#designSettingsModal .nav-link {
    color: #495057;
    border: none;
    border-bottom: 3px solid transparent;
    padding: 0.5rem 1rem;
    font-size: 0.9rem;
}

#designSettingsModal .nav-link:hover {
    border-bottom-color: #dee2e6;
}

#designSettingsModal .nav-link.active {
    color: #007bff;
    background-color: transparent;
    border-bottom-color: #007bff;
}

#designSettingsModal .form-control-color {
    width: 100%;
    height: 40px;
    padding: 0.375rem;
}

#designSettingsModal .card {
    box-shadow: 0 0 10px rgba(0,0,0,0.1);
}

#headingPreview h1,
#headingPreview h2,
#headingPreview h3 {
    margin-bottom: 1rem;
}

/* TinyMCEコンテナのスタイル調整 */
.tox-tinymce {
    border-radius: 0.25rem;
}

/* アニメーションパターンのプレビュー用CSS */
@keyframes float {
    0%, 100% {
        transform: translateY(0px);
    }
    50% {
        transform: translateY(-20px);
    }
}

@keyframes wave {
    0%, 100% {
        transform: translateY(0);
    }
    50% {
        transform: translateY(-10px);
    }
}

@keyframes rotate {
    from {
        transform: rotate(0deg);
    }
    to {
        transform: rotate(360deg);
    }
}

@keyframes twinkle {
    0%, 100% {
        opacity: 0.3;
    }
    50% {
        opacity: 1;
    }
}

/* 新しいアニメーションパターン */
@keyframes aurora {
    0% {
        transform: translateX(-50%);
    }
    100% {
        transform: translateX(0%);
    }
}

@keyframes matrix-fall {
    0% {
        transform: translateY(-20px);
        opacity: 0;
    }
    10% {
        opacity: 1;
    }
    90% {
        opacity: 1;
    }
    100% {
        transform: translateY(320px);
        opacity: 0;
    }
}

@keyframes bubble-rise {
    0% {
        transform: translateY(0) translateX(0);
        opacity: 0;
    }
    10% {
        opacity: 0.6;
    }
    50% {
        transform: translateY(-150px) translateX(20px);
    }
    100% {
        transform: translateY(-350px) translateX(-20px);
        opacity: 0;
    }
}

@keyframes pulse {
    0%, 100% {
        transform: scale(1);
        opacity: 0.2;
    }
    50% {
        transform: scale(1.1);
        opacity: 0.4;
    }
}

@keyframes circuit-flow {
    0% {
        background: linear-gradient(90deg, transparent 0%, var(--pattern-color, #667eea) 50%, transparent 100%);
        background-size: 200% 100%;
        background-position: -100% 0;
    }
    100% {
        background-position: 100% 0;
    }
}

@keyframes circuit-flow-vertical {
    0% {
        background: linear-gradient(180deg, transparent 0%, var(--pattern-color, #667eea) 50%, transparent 100%);
        background-size: 100% 200%;
        background-position: 0 -100%;
    }
    100% {
        background-position: 0 100%;
    }
}

@keyframes galaxy-rotate {
    0% {
        transform: rotate(0deg) scale(1);
    }
    50% {
        transform: rotate(180deg) scale(1.2);
    }
    100% {
        transform: rotate(360deg) scale(1);
    }
}

@keyframes orbit {
    0% {
        transform: rotate(0deg) translateX(50px) rotate(0deg);
    }
    100% {
        transform: rotate(360deg) translateX(50px) rotate(-360deg);
    }
}

/* 新しいアニメーション5種類 */
@keyframes rain-fall {
    0% {
        transform: translateY(-20px);
        opacity: 0;
    }
    10% {
        opacity: 1;
    }
    90% {
        opacity: 1;
    }
    100% {
        transform: translateY(320px);
        opacity: 0;
    }
}

@keyframes fire-flicker {
    0%, 100% {
        transform: scaleY(1) translateY(0);
        opacity: 0.8;
    }
    25% {
        transform: scaleY(1.1) translateY(-5px);
        opacity: 0.9;
    }
    50% {
        transform: scaleY(0.9) translateY(2px);
        opacity: 0.7;
    }
    75% {
        transform: scaleY(1.05) translateY(-3px);
        opacity: 0.85;
    }
}

@keyframes fire-dance {
    0%, 100% {
        transform: translateX(0) scaleX(1);
    }
    25% {
        transform: translateX(-10px) scaleX(1.1);
    }
    50% {
        transform: translateX(5px) scaleX(0.9);
    }
    75% {
        transform: translateX(-5px) scaleX(1.05);
    }
}

@keyframes snow-fall {
    0% {
        transform: translateY(-10px) translateX(0);
        opacity: 0;
    }
    10% {
        opacity: 1;
    }
    100% {
        transform: translateY(320px) translateX(20px);
        opacity: 0;
    }
}

@keyframes mosaic-fade {
    0%, 100% {
        opacity: 0.2;
        transform: scale(1);
    }
    50% {
        opacity: 0.8;
        transform: scale(1.1);
    }
}

@keyframes sparkle {
    0%, 100% {
        opacity: 0;
        transform: scale(0);
    }
    50% {
        opacity: 1;
        transform: scale(1);
    }
}

@keyframes crystal-rotate {
    0% {
        transform: rotate(45deg) scale(1);
        opacity: 0.6;
    }
    50% {
        transform: rotate(225deg) scale(1.2);
        opacity: 1;
    }
    100% {
        transform: rotate(405deg) scale(1);
        opacity: 0.6;
    }
}

@keyframes sakura-fall {
    0% {
        transform: translateY(-20px) translateX(0) rotate(0deg);
        opacity: 0;
    }
    10% {
        opacity: 1;
    }
    100% {
        transform: translateY(320px) translateX(50px) rotate(360deg);
        opacity: 0;
    }
}

@keyframes firefly-move {
    0%, 100% {
        transform: translate(0, 0);
        opacity: 0.2;
    }
    25% {
        transform: translate(30px, -20px);
        opacity: 1;
    }
    50% {
        transform: translate(-20px, -40px);
        opacity: 0.8;
    }
    75% {
        transform: translate(40px, 10px);
        opacity: 1;
    }
}

@keyframes rainbow-wave {
    0% {
        transform: translateX(-100%);
    }
    100% {
        transform: translateX(100%);
    }
}

@keyframes diamond-shine {
    0%, 100% {
        transform: rotate(45deg) scale(1);
        opacity: 0.5;
        filter: brightness(1);
    }
    50% {
        transform: rotate(45deg) scale(1.2);
        opacity: 1;
        filter: brightness(1.5);
    }
}

@keyframes glitter-twinkle {
    0%, 100% {
        opacity: 0;
        transform: scale(0);
    }
    50% {
        opacity: 1;
        transform: scale(1.5);
    }
}

@keyframes stardust-drift {
    0% {
        transform: translate(0, 0) rotate(0deg);
    }
    100% {
        transform: translate(100px, -100px) rotate(360deg);
    }
}

@keyframes star-pulse {
    0%, 100% {
        transform: scale(1);
        opacity: 0.5;
    }
    50% {
        transform: scale(1.5);
        opacity: 1;
    }
}

/* 新しい5つの派手な背景アニメーション */
@keyframes cyberRain {
    0% {
        background-position: 0% 0%;
    }
    100% {
        background-position: 0% 100%;
    }
}

@keyframes holographicWave {
    0% {
        background-position: 0% 50%;
        filter: hue-rotate(0deg);
    }
    50% {
        background-position: 100% 50%;
        filter: hue-rotate(180deg);
    }
    100% {
        background-position: 0% 50%;
        filter: hue-rotate(360deg);
    }
}

@keyframes plasmaStorm {
    0%, 100% {
        transform: scale(1) rotate(0deg);
        filter: blur(0px);
    }
    25% {
        transform: scale(1.1) rotate(90deg);
        filter: blur(1px);
    }
    50% {
        transform: scale(0.9) rotate(180deg);
        filter: blur(2px);
    }
    75% {
        transform: scale(1.05) rotate(270deg);
        filter: blur(1px);
    }
}

@keyframes quantumField {
    0% {
        transform: rotate(0deg);
        filter: hue-rotate(0deg) brightness(1);
    }
    50% {
        filter: hue-rotate(180deg) brightness(1.2);
    }
    100% {
        transform: rotate(360deg);
        filter: hue-rotate(360deg) brightness(1);
    }
}

@keyframes nebulaCloud {
    0%, 100% {
        transform: scale(1) translateY(0);
        opacity: 0.8;
    }
    33% {
        transform: scale(1.2) translateY(-20px);
        opacity: 1;
    }
    66% {
        transform: scale(0.8) translateY(20px);
        opacity: 0.6;
    }
}

/* フリースペースの文字列改行処理 - 絶対にはみ出さない */
.free-space,
textarea.free-space,
[id*="FreeSpace"],
.form-control[id*="FreeSpace"] {
    word-wrap: break-word !important;
    word-break: break-all !important;
    overflow-wrap: anywhere !important;
    max-width: 100% !important;
    overflow-x: auto !important;
    white-space: pre-wrap !important;
    hyphens: auto !important;
}
</style>