<?php
// 履歴データの読み込み
$historyFile = __DIR__ . '/../../data/history.json';
$dugaHistoryFile = __DIR__ . '/../../data/duga-history.json';
$history = [];

// 通常の履歴を読み込み
$normalHistoryCount = 0;
if (file_exists($historyFile)) {
    $normalHistory = json_decode(file_get_contents($historyFile), true) ?: [];
    $normalHistoryCount = count($normalHistory);
    $history = $normalHistory;
}

// DUGA履歴を読み込み
$dugaHistoryCount = 0;
if (file_exists($dugaHistoryFile)) {
    $dugaHistory = json_decode(file_get_contents($dugaHistoryFile), true) ?: [];
    $dugaHistoryCount = count($dugaHistory);
    // 履歴をマージ（created_atで並び替え）
    $history = array_merge($history, $dugaHistory);
    
    // created_atでソート（新しい順）
    usort($history, function($a, $b) {
        $timeA = strtotime($a['created_at'] ?? '0');
        $timeB = strtotime($b['created_at'] ?? '0');
        return $timeB - $timeA;
    });
}

// デバッグ情報（本番環境では削除）
// error_log("History loaded - Normal: $normalHistoryCount, DUGA: $dugaHistoryCount, Total: " . count($history));

// サイト設定の読み込み
$siteConfigFile = __DIR__ . '/../../data/site-configs.json';
$siteConfigs = [];
if (file_exists($siteConfigFile)) {
    $siteConfigs = json_decode(file_get_contents($siteConfigFile), true) ?: [];
}

// 各記事がどのサイトに属しているかをマッピング
$articleSiteMap = [];
foreach ($siteConfigs as $configId => $config) {
    if (isset($config['article_ids'])) {
        foreach ($config['article_ids'] as $articleId) {
            $articleSiteMap[$articleId] = $configId;
        }
    }
}

// 検索処理
$searchQuery = isset($_GET['search']) ? trim($_GET['search']) : '';
$categoryFilter = isset($_GET['category']) ? trim($_GET['category']) : '';

// 検索とフィルタリング
$filteredHistory = $history;
if (!empty($searchQuery) || !empty($categoryFilter)) {
    $filteredHistory = array_filter($history, function($item) use ($searchQuery, $categoryFilter) {
        $matchSearch = true;
        $matchCategory = true;
        
        // 検索クエリでフィルタリング
        if (!empty($searchQuery)) {
            $searchLower = mb_strtolower($searchQuery, 'UTF-8');
            $titleMatch = mb_strpos(mb_strtolower($item['title'] ?? '', 'UTF-8'), $searchLower) !== false;
            $contentMatch = mb_strpos(mb_strtolower($item['content'] ?? '', 'UTF-8'), $searchLower) !== false;
            $slugMatch = mb_strpos(mb_strtolower($item['slug'] ?? '', 'UTF-8'), $searchLower) !== false;
            $matchSearch = $titleMatch || $contentMatch || $slugMatch;
        }
        
        // カテゴリでフィルタリング
        if (!empty($categoryFilter)) {
            $matchCategory = ($item['category'] ?? '') === $categoryFilter;
        }
        
        return $matchSearch && $matchCategory;
    });
    // array_filterは配列のキーを保持するため、array_valuesで再インデックス
    $filteredHistory = array_values($filteredHistory);
}

// カテゴリ一覧を取得（履歴から収集）
$allCategories = ['ビジネス', 'テクノロジー', 'ライフスタイル', '健康・美容', 'エンタメ', '教育', '金融', '旅行', 'グルメ', 'その他'];
$customCategories = [];

// 履歴からカテゴリを収集
foreach ($history as $item) {
    $category = $item['category'] ?? '';
    if (!empty($category) && !in_array($category, $allCategories) && !in_array($category, $customCategories)) {
        $customCategories[] = $category;
    }
}

// カスタムカテゴリをソートして追加
sort($customCategories);
$allCategories = array_merge($allCategories, $customCategories);

// ページネーション
$perPage = 20;
$page = isset($_GET['p']) ? max(1, intval($_GET['p'])) : 1;
$totalItems = count($filteredHistory);
$totalPages = ceil($totalItems / $perPage);
$offset = ($page - 1) * $perPage;
$currentPageItems = array_slice($filteredHistory, $offset, $perPage);
?>
<div class="history-page">
<style>
/* 履歴ページの幅を修正 */
.history-page {
    max-width: 100%;
    width: 100%;
}

/* 検索カードの高さ制限 */
.history-page .search-card {
    max-width: 100%;
    width: 100%;
}

.history-page .search-card .card-body {
    padding: 0.75rem !important;
}

.history-page .search-card .d-flex {
    align-items: center;
    gap: 0.5rem;
}

.history-page .search-card input,
.history-page .search-card select {
    height: 32px !important;
    padding: 0.25rem 0.5rem !important;
    font-size: 0.875rem !important;
}

.history-page .search-card .btn {
    height: 32px !important;
    padding: 0.25rem 0.75rem !important;
    display: inline-flex;
    align-items: center;
}

/* テーブルカードの調整 */
.history-page .table-card {
    max-width: 100%;
    width: 100%;
}

.history-page .table-responsive {
    max-width: 100%;
    overflow-x: auto;
}

/* テーブルのコンパクト化 */
.history-page table.table-sm td,
.history-page table.table-sm th {
    padding: 0.25rem !important;
}

/* バッジのサイズ調整 */
.history-page .badge {
    font-size: 0.75rem !important;
    padding: 0.2rem 0.4rem !important;
}

/* 操作ボタンのサイズ調整 */
.history-page .btn-sm {
    padding: 0.2rem 0.4rem !important;
    font-size: 0.8rem !important;
}
</style>
<div class="d-flex justify-content-between align-items-center mb-2">
    <h1 class="h5 mb-0"><i class="bi bi-clock-history"></i> 生成履歴 <span class="badge bg-secondary small"><?php echo count($history); ?>件</span></h1>
</div>

<!-- 検索フォーム -->
<div class="card mb-2 search-card" style="width: 100%; max-width: 100%; box-sizing: border-box;">
    <div class="card-body p-2">
        <form method="get" action="">
            <input type="hidden" name="page" value="history">
            <div class="d-flex gap-2" style="max-width: 100%;">
                <input type="text" class="form-control form-control-sm" id="searchInput" name="search" 
                       value="<?php echo htmlspecialchars($searchQuery); ?>" 
                       placeholder="検索..." style="max-width: 200px;">
                <select class="form-select form-select-sm" id="categoryFilter" name="category" style="max-width: 150px;">
                    <option value="">全カテゴリ</option>
                    <?php foreach ($allCategories as $cat): ?>
                        <option value="<?php echo htmlspecialchars($cat); ?>" 
                                <?php echo $categoryFilter === $cat ? 'selected' : ''; ?>>
                            <?php echo htmlspecialchars($cat); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                <div>
                    <button type="submit" class="btn btn-primary btn-sm">
                        <i class="bi bi-search"></i>
                    </button>
                    <?php if (!empty($searchQuery) || !empty($categoryFilter)): ?>
                        <a href="?page=history" class="btn btn-secondary btn-sm">
                            <i class="bi bi-x-circle"></i>
                        </a>
                    <?php endif; ?>
                </div>
            </div>
        </form>
        
        <?php if (!empty($searchQuery) || !empty($categoryFilter)): ?>
            <div class="mt-2">
                <small class="text-muted">
                    <?php echo count($filteredHistory); ?>件
                    <?php if (!empty($searchQuery)): ?>
                        - <?php echo htmlspecialchars($searchQuery); ?>
                    <?php endif; ?>
                </small>
            </div>
        <?php endif; ?>
    </div>
</div>

<div class="card table-card" style="width: 100%; max-width: 100%; box-sizing: border-box;">
    <div class="card-body">
        <?php if (empty($filteredHistory)): ?>
            <p class="text-center text-muted py-5">まだ記事が生成されていません。</p>
        <?php else: ?>
            <div class="table-responsive">
                <table class="table table-hover table-sm">
                    <thead>
                        <tr>
                            <th class="small">タイトル</th>
                            <th class="small">URLスラッグ</th>
                            <th class="small">カテゴリー</th>
                            <th class="small">サイト</th>
                            <th class="small">モデル</th>
                            <th class="small">画像</th>
                            <th class="small">文字数</th>
                            <th class="small">生成日時</th>
                            <th class="small" style="width: 200px;">操作</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($currentPageItems as $item): ?>
                            <tr>
                                <td class="text-truncate small py-1" style="max-width: 200px;">
                                    <?php echo htmlspecialchars($item['title'] ?? '無題'); ?>
                                </td>
                                <td class="text-truncate small py-1" style="max-width: 150px;">
                                    <?php 
                                    // URLスラッグを生成（static-site-generatorと同じロジック）
                                    $slug = '';
                                    if (!empty($item['slug'])) {
                                        // カスタムslugが設定されている場合
                                        $slug = $item['slug'];
                                    } else {
                                        // slugが空の場合は記事IDを使用
                                        $slug = $item['id'] ?? '';
                                    }
                                    
                                    // 日本語の場合はデコードして表示
                                    $displaySlug = urldecode($slug);
                                    ?>
                                    <code><?php echo htmlspecialchars($displaySlug); ?>.html</code>
                                </td>
                                <td>
                                    <span class="badge bg-info editable-category" 
                                          data-article-id="<?php echo htmlspecialchars($item['id']); ?>"
                                          style="cursor: pointer;">
                                        <?php echo htmlspecialchars($item['category'] ?? 'その他'); ?>
                                    </span>
                                </td>
                                <td>
                                    <?php 
                                    $articleId = $item['id'];
                                    $siteConfigId = $articleSiteMap[$articleId] ?? null;
                                    if ($siteConfigId && isset($siteConfigs[$siteConfigId])) {
                                        echo '<span class="badge bg-success">' . htmlspecialchars($siteConfigs[$siteConfigId]['display_name']) . '</span>';
                                    } else {
                                        echo '<span class="badge bg-secondary editable-site" data-article-id="' . htmlspecialchars($articleId) . '" style="cursor: pointer;">未設定</span>';
                                    }
                                    ?>
                                </td>
                                <td>
                                    <span class="badge bg-secondary">
                                        <?php 
                                        // モデル名をそのまま表示（config.phpは不要）
                                        $modelName = $item['model'] ?? 'Unknown';
                                        
                                        // モデル名の整形（必要に応じて）
                                        $displayNames = [
                                            'gpt-4o-mini' => 'GPT-4o Mini',
                                            'gpt-4o' => 'GPT-4o',
                                            'gpt-3.5-turbo' => 'GPT-3.5 Turbo',
                                            'claude-3-haiku' => 'Claude 3 Haiku',
                                            'claude-3-sonnet' => 'Claude 3 Sonnet',
                                            'claude-3-5-sonnet' => 'Claude 3.5 Sonnet',
                                            'claude-3-opus' => 'Claude 3 Opus',
                                            'gemini-1.5-flash' => 'Gemini 1.5 Flash',
                                            'gemini-1.5-flash-8b' => 'Gemini 1.5 Flash 8B',
                                            'gemini-1.5-pro' => 'Gemini 1.5 Pro',
                                            'llama-3.2-sonar-128k-online' => 'Perplexity Online',
                                            'llama-3.1-sonar-128k-online' => 'Perplexity Sonar',
                                            'llama-3.3-70b' => 'Llama 3.3 70B',
                                            'llama-3.1-405b' => 'Llama 3.1 405B',
                                            'venice-gpt-4o' => 'Venice GPT-4o',
                                            'venice-claude-3-5-sonnet' => 'Venice Claude 3.5'
                                        ];
                                        
                                        $displayName = $displayNames[$modelName] ?? $modelName;
                                        echo htmlspecialchars($displayName);
                                        ?>
                                    </span>
                                </td>
                                <td>
                                    <?php 
                                    // 通常の画像数をカウント
                                    $imageCount = count($item['images'] ?? []);
                                    
                                    // DMM記事の場合、サンプル画像数も含める
                                    if (isset($item['metadata']['type']) && $item['metadata']['type'] === 'dmm') {
                                        $dmmImageCount = count($item['metadata']['dmm_data']['sample_images'] ?? []);
                                        if (!empty($item['metadata']['dmm_data']['product_image'])) {
                                            $dmmImageCount++;
                                        }
                                        $imageCount += $dmmImageCount;
                                    }
                                    
                                    // DUGA記事の場合、サンプル画像数も含める
                                    if (isset($item['metadata']['type']) && $item['metadata']['type'] === 'duga') {
                                        $dugaImageCount = count($item['metadata']['duga_data']['sample_images'] ?? []);
                                        if (!empty($item['metadata']['duga_data']['product_image'])) {
                                            $dugaImageCount++;
                                        }
                                        if (!empty($item['metadata']['duga_data']['jacket_image'])) {
                                            $dugaImageCount++;
                                        }
                                        $imageCount += $dugaImageCount;
                                    }
                                    
                                    if ($imageCount > 0): 
                                    ?>
                                        <span class="badge bg-success">
                                            <i class="bi bi-image"></i> <?php echo $imageCount; ?>
                                        </span>
                                    <?php else: ?>
                                        <span class="text-muted">-</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php 
                                    $charCount = mb_strlen(strip_tags($item['content'] ?? ''));
                                    echo number_format($charCount) . '文字';
                                    ?>
                                </td>
                                <td><?php echo date('Y/m/d H:i', strtotime($item['created_at'])); ?></td>
                                <td>
                                    <button type="button" class="btn btn-sm btn-outline-primary" 
                                            onclick="viewArticle('<?php echo htmlspecialchars($item['id']); ?>')"
                                            title="記事を表示">
                                        <i class="bi bi-eye"></i>
                                    </button>
                                    <button type="button" class="btn btn-sm btn-outline-success" 
                                            onclick="editArticle('<?php echo htmlspecialchars($item['id']); ?>')"
                                            title="編集">
                                        <i class="bi bi-pencil"></i>
                                    </button>
                                    <button type="button" class="btn btn-sm btn-outline-danger" 
                                            onclick="deleteArticle('<?php echo htmlspecialchars($item['id']); ?>')"
                                            title="削除">
                                        <i class="bi bi-trash"></i>
                                    </button>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            
            <?php if ($totalPages > 1): ?>
                <nav aria-label="Page navigation" class="mt-4">
                    <ul class="pagination justify-content-center">
                        <?php 
                        $queryParams = [];
                        if (!empty($searchQuery)) $queryParams['search'] = $searchQuery;
                        if (!empty($categoryFilter)) $queryParams['category'] = $categoryFilter;
                        $queryString = !empty($queryParams) ? '&' . http_build_query($queryParams) : '';
                        ?>
                        <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                            <a class="page-link" href="?page=history&p=<?php echo $page - 1; ?><?php echo $queryString; ?>">前へ</a>
                        </li>
                        <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                            <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                                <a class="page-link" href="?page=history&p=<?php echo $i; ?><?php echo $queryString; ?>"><?php echo $i; ?></a>
                            </li>
                        <?php endfor; ?>
                        <li class="page-item <?php echo $page >= $totalPages ? 'disabled' : ''; ?>">
                            <a class="page-link" href="?page=history&p=<?php echo $page + 1; ?><?php echo $queryString; ?>">次へ</a>
                        </li>
                    </ul>
                </nav>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</div>

<!-- カテゴリー編集モーダル -->
<div class="modal fade" id="categoryEditModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">カテゴリーを編集</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="mb-3">
                    <label class="form-label">既存のカテゴリーから選択</label>
                    <select class="form-control" id="categorySelect">
                        <option value="">-- 選択してください --</option>
                        <?php 
                        $categories = array_unique(array_filter(array_column($history, 'category')));
                        sort($categories);
                        foreach ($categories as $cat): 
                        ?>
                        <option value="<?php echo htmlspecialchars($cat); ?>"><?php echo htmlspecialchars($cat); ?></option>
                        <?php endforeach; ?>
                        <option value="__new__">➕ 新規カテゴリー</option>
                    </select>
                </div>
                <div class="mb-3" id="newCategoryDiv" style="display: none;">
                    <label class="form-label">新しいカテゴリー名</label>
                    <input type="text" class="form-control" id="newCategoryInput" placeholder="カテゴリー名を入力">
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">キャンセル</button>
                <button type="button" class="btn btn-primary" id="saveCategoryBtn">保存</button>
            </div>
        </div>
    </div>
</div>

<!-- サイト割り当てモーダル -->
<div class="modal fade" id="siteAssignModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">サイトに割り当て</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="alert alert-info">
                    <i class="bi bi-info-circle"></i> この記事を含めるサイトを選択してください。
                </div>
                <div class="list-group" id="siteList">
                    <?php foreach ($siteConfigs as $configId => $config): ?>
                    <label class="list-group-item">
                        <input class="form-check-input me-2" type="checkbox" 
                               value="<?php echo htmlspecialchars($configId); ?>"
                               data-site-name="<?php echo htmlspecialchars($config['display_name']); ?>">
                        <strong><?php echo htmlspecialchars($config['display_name']); ?></strong>
                        <br>
                        <small class="text-muted"><?php echo htmlspecialchars($config['url']); ?></small>
                    </label>
                    <?php endforeach; ?>
                </div>
                <?php if (empty($siteConfigs)): ?>
                <p class="text-muted text-center py-3">サイトがまだ作成されていません。</p>
                <div class="text-center">
                    <a href="?page=site-generator" class="btn btn-primary">サイトを作成</a>
                </div>
                <?php endif; ?>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">キャンセル</button>
                <button type="button" class="btn btn-primary" id="saveSiteAssignBtn">保存</button>
            </div>
        </div>
    </div>
</div>

<script>
// 履歴から記事を表示
function viewArticle(articleId) {
    // 履歴データから該当記事を取得
    fetch('../api/service.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({
            action: 'get_history'
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            const article = data.history.find(item => item.id === articleId);
            if (article) {
                // 記事をグローバル変数に保存
                window.currentArticle = article.content;
                window.currentImages = article.images || [];
                
                // 履歴モーダルを表示
                showHistoryModal(article);
            }
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('記事の読み込みに失敗しました');
    });
}

// 履歴からSNS投稿を作成
function createSnsFromHistory(articleId) {
    // 履歴データから該当記事を取得
    fetch('../api/service.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({
            action: 'get_history'
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            const article = data.history.find(item => item.id === articleId);
            if (article) {
                // 記事をグローバル変数に保存
                window.currentArticle = article.content;
                window.currentImages = article.images || [];
                
                // SNS投稿モーダルを表示
                if (typeof showSnsModal === 'function') {
                    showSnsModal();
                } else {
                    console.error('showSnsModal function not found');
                    alert('SNS投稿機能の読み込みエラーが発生しました。ページを再読み込みしてください。');
                }
            }
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('記事の読み込みに失敗しました');
    });
}

// 記事を削除
function deleteArticle(articleId) {
    if (!confirm('この記事を削除してもよろしいですか？')) {
        return;
    }
    
    fetch('../api/service.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({
            action: 'delete_history',
            id: articleId
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload();
        } else {
            alert('削除に失敗しました: ' + (data.error || '不明なエラー'));
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('削除に失敗しました');
    });
}

// 履歴モーダル表示
function showHistoryModal(article) {
    const modalHtml = `
        <div class="modal fade" id="historyModal" tabindex="-1">
            <div class="modal-dialog modal-lg">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">${article.title || '無題'}</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body">
                        <div id="historyContent">${article.content}</div>
                        ${article.images && article.images.length > 0 ? `
                            <hr>
                            <h6>画像一覧</h6>
                            <div class="row">
                                ${article.images.map(img => `
                                    <div class="col-md-4 mb-3">
                                        <img src="${img.url}" alt="${img.alt}" class="img-fluid rounded">
                                        <small class="text-muted d-block mt-1">${img.alt}</small>
                                    </div>
                                `).join('')}
                            </div>
                        ` : ''}
                        ${article.metadata && article.metadata.type === 'dmm' && article.metadata.dmm_data && article.metadata.dmm_data.sample_images && article.metadata.dmm_data.sample_images.length > 0 ? `
                            <hr>
                            <h6>商品サンプル画像</h6>
                            <div class="row">
                                ${article.metadata.dmm_data.sample_images.map((img, idx) => `
                                    <div class="col-md-4 mb-3">
                                        <img src="${img}" alt="サンプル画像${idx + 1}" class="img-fluid rounded" style="cursor: pointer;" onclick="window.open('${img}', '_blank')">
                                        <small class="text-muted d-block mt-1">サンプル画像${idx + 1}</small>
                                    </div>
                                `).join('')}
                            </div>
                        ` : ''}
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-info" id="snsButtonInModal">
                            <i class="bi bi-share"></i> SNS投稿を作成
                        </button>
                        <button type="button" class="btn btn-secondary" onclick="copyArticleContent()">
                            <i class="bi bi-clipboard"></i> コピー
                        </button>
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">閉じる</button>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    // 既存のモーダルを削除
    const existingModal = document.getElementById('historyModal');
    if (existingModal) {
        existingModal.remove();
    }
    
    // モーダルを追加して表示
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    const modal = new bootstrap.Modal(document.getElementById('historyModal'));
    modal.show();
    
    // SNSボタンのイベントハンドラを設定
    document.getElementById('snsButtonInModal').addEventListener('click', function() {
        // 記事内容を確実に設定
        window.currentArticle = article.content;
        
        // showSnsModal関数の存在確認
        if (typeof showSnsModal === 'function') {
            showSnsModal();
        } else {
            console.error('showSnsModal function not found');
            alert('SNS投稿機能の読み込みエラーが発生しました。ページを再読み込みしてください。');
        }
    });
}

// 記事内容をコピー
async function copyArticleContent() {
    const content = document.getElementById('historyContent')?.innerHTML || '';
    const text = stripTags(content);
    
    try {
        if (navigator.clipboard && navigator.clipboard.writeText) {
            await navigator.clipboard.writeText(text);
            showToast('クリップボードにコピーしました', 'success');
        } else {
            const textarea = document.createElement('textarea');
            textarea.value = text;
            textarea.style.position = 'fixed';
            textarea.style.opacity = '0';
            document.body.appendChild(textarea);
            textarea.select();
            document.execCommand('copy');
            document.body.removeChild(textarea);
            showToast('クリップボードにコピーしました', 'success');
        }
    } catch (error) {
        showError('コピーに失敗しました: ' + error.message);
    }
}

// ユーティリティ関数
function stripTags(html) {
    const tmp = document.createElement('div');
    tmp.innerHTML = html;
    return tmp.textContent || tmp.innerText || '';
}

function showToast(message, type) {
    // トーストメッセージの表示
    const toastHtml = `
        <div class="toast align-items-center text-white bg-${type === 'success' ? 'success' : 'danger'} border-0" 
             role="alert" aria-live="assertive" aria-atomic="true" style="position: fixed; top: 20px; right: 20px; z-index: 9999;">
            <div class="d-flex">
                <div class="toast-body">
                    ${message}
                </div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
            </div>
        </div>
    `;
    
    const toastContainer = document.createElement('div');
    toastContainer.innerHTML = toastHtml;
    document.body.appendChild(toastContainer);
    
    const toastElement = toastContainer.querySelector('.toast');
    const toast = new bootstrap.Toast(toastElement);
    toast.show();
    
    // 5秒後に削除
    setTimeout(() => {
        toastContainer.remove();
    }, 5000);
}

function showError(message) {
    alert('エラー: ' + message);
}

// 記事編集ページへ遷移
function editArticle(articleId) {
    window.location.href = '?page=editor&id=' + articleId;
}

// カテゴリー編集
let currentEditArticleId = null;

$(document).on('click', '.editable-category', function() {
    currentEditArticleId = $(this).data('article-id');
    const currentCategory = $(this).text();
    
    $('#categorySelect').val(currentCategory);
    $('#newCategoryDiv').hide();
    
    const modal = new bootstrap.Modal(document.getElementById('categoryEditModal'));
    modal.show();
});

$('#categorySelect').change(function() {
    if ($(this).val() === '__new__') {
        $('#newCategoryDiv').show();
        $('#newCategoryInput').focus();
    } else {
        $('#newCategoryDiv').hide();
    }
});

$('#saveCategoryBtn').click(function() {
    let category = $('#categorySelect').val();
    if (category === '__new__') {
        category = $('#newCategoryInput').val().trim();
        if (!category) {
            alert('カテゴリー名を入力してください');
            return;
        }
    }
    
    if (!category) {
        alert('カテゴリーを選択してください');
        return;
    }
    
    // APIで更新
    $.ajax({
        url: '../api/update-article-category.php',
        method: 'POST',
        data: {
            article_id: currentEditArticleId,
            category: category
        },
        success: function(response) {
            if (response.success) {
                location.reload();
            } else {
                alert('エラー: ' + response.error);
            }
        },
        error: function() {
            alert('通信エラーが発生しました');
        }
    });
});

// サイト割り当て
$(document).on('click', '.editable-site', function() {
    currentEditArticleId = $(this).data('article-id');
    
    // 現在の割り当て状態をチェックボックスに反映
    $('#siteList input[type="checkbox"]').prop('checked', false);
    
    <?php foreach ($siteConfigs as $configId => $config): ?>
    <?php if (isset($config['article_ids'])): ?>
    if (<?php echo json_encode($config['article_ids']); ?>.includes(currentEditArticleId)) {
        $('#siteList input[value="<?php echo $configId; ?>"]').prop('checked', true);
    }
    <?php endif; ?>
    <?php endforeach; ?>
    
    const modal = new bootstrap.Modal(document.getElementById('siteAssignModal'));
    modal.show();
});

$('#saveSiteAssignBtn').click(function() {
    const selectedSites = [];
    $('#siteList input[type="checkbox"]:checked').each(function() {
        selectedSites.push($(this).val());
    });
    
    // APIで更新
    $.ajax({
        url: '../api/update-article-sites.php',
        method: 'POST',
        data: {
            article_id: currentEditArticleId,
            site_ids: selectedSites
        },
        success: function(response) {
            if (response.success) {
                location.reload();
            } else {
                alert('エラー: ' + response.error);
            }
        },
        error: function() {
            alert('通信エラーが発生しました');
        }
    });
})
</script>
</div> <!-- /.history-page -->