<?php
// プリセットの読み込み
$presetsFile = __DIR__ . '/../../data/presets.json';
$presets = [];
if (file_exists($presetsFile)) {
    $presets = json_decode(file_get_contents($presetsFile), true) ?: [];
}

// 記事データからカテゴリー一覧を取得
$historyFile = __DIR__ . '/../../data/history.json';
$categories = [];
if (file_exists($historyFile)) {
    $history = json_decode(file_get_contents($historyFile), true) ?: [];
    foreach ($history as $article) {
        $category = $article['category'] ?? 'その他';
        if (!in_array($category, $categories)) {
            $categories[] = $category;
        }
    }
    sort($categories);
}

// デフォルトカテゴリー
$defaultCategories = ['ビジネス', 'テクノロジー', 'ライフスタイル', '健康・美容', 'エンタメ', '教育', '金融', '旅行', 'グルメ', 'その他'];

// 全カテゴリーをマージして重複を削除
$allCategories = array_unique(array_merge($defaultCategories, $categories));
?>

<div class="page-header">
    <h1><i class="bi bi-file-text"></i> 記事生成</h1>
    <p class="text-muted">AIを使用して高品質な記事を生成します</p>
</div>

<div class="row">
    <div class="col-md-8">
        <div class="card">
            <div class="card-body">
                <form id="generateForm" method="POST" action="">
                    <!-- プリセット選択 -->
                    <div class="mb-4">
                        <label for="preset" class="form-label">プリセット</label>
                        <select class="form-control" id="preset" onchange="loadPreset()">
                            <option value="">-- プリセットを選択 --</option>
                            <?php foreach ($presets as $key => $preset): ?>
                                <option value="<?php echo htmlspecialchars($key); ?>">
                                    <?php echo htmlspecialchars($preset['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <!-- 基本設定 -->
                    <div class="mb-4">
                        <label for="title" class="form-label">記事タイトル（AIが魅力的に改善します）</label>
                        <input type="text" class="form-control" id="title" 
                               placeholder="例：AIの最新トレンド2025" required>
                    </div>
                    
                    <div class="mb-4">
                        <label for="keywords" class="form-label">キーワード</label>
                        <input type="text" class="form-control" id="keywords" 
                               placeholder="カンマ区切りで入力（例：AI, 人工知能, 機械学習）">
                    </div>
                    
                    <div class="mb-4">
                        <label for="category" class="form-label">カテゴリー</label>
                        <div id="category_selector">
                            <div id="select_mode">
                                <select class="form-control" id="category" onchange="toggleCustomCategory()">
                                    <option value="">-- カテゴリーを選択 --</option>
                                    <?php foreach ($allCategories as $cat): ?>
                                    <option value="<?php echo htmlspecialchars($cat); ?>"><?php echo htmlspecialchars($cat); ?></option>
                                    <?php endforeach; ?>
                                    <option value="__new__" style="color: #28a745; font-weight: bold;">➕ 新規カテゴリーを追加</option>
                                </select>
                            </div>
                            <div id="custom_mode" style="display: none;">
                                <div class="alert alert-success p-2 mb-2">
                                    <i class="bi bi-plus-circle"></i> 新しいカテゴリーを作成
                                </div>
                                <div class="input-group">
                                    <input type="text" class="form-control" id="custom_category" 
                                           placeholder="新しいカテゴリー名を入力" autofocus>
                                    <button type="button" class="btn btn-outline-secondary" onclick="cancelCustomCategory()">
                                        <i class="bi bi-x"></i> キャンセル
                                    </button>
                                </div>
                            </div>
                        </div>
                        <small class="form-text text-muted" id="category_hint">
                            <i class="bi bi-info-circle"></i> 既存のカテゴリーを選択するか、「新規カテゴリーを追加」を選んで作成できます
                        </small>
                    </div>
                    
                    
                    <div class="row mb-4">
                        <div class="col-md-6">
                            <label for="model" class="form-label">AIモデル</label>
                            <select class="form-control" id="model">
                                <?php foreach ($config['models'] as $provider => $models): ?>
                                    <optgroup label="<?php echo ucfirst($provider); ?>">
                                        <?php foreach ($models as $key => $model): ?>
                                            <option value="<?php echo htmlspecialchars($key); ?>" 
                                                    <?php echo $key === $config['default_model'] ? 'selected' : ''; ?>>
                                                <?php echo htmlspecialchars($model['name']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </optgroup>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-6">
                            <label for="output_format" class="form-label">出力形式</label>
                            <select class="form-control" id="output_format">
                                <option value="visual">ビジュアルエディタ</option>
                                <option value="html">HTML</option>
                                <option value="markdown">Markdown</option>
                                <option value="text">プレーンテキスト</option>
                            </select>
                        </div>
                    </div>
                    
                    <!-- 詳細設定 -->
                    <div class="mb-4">
                        <label for="system_prompt" class="form-label">システムプロンプト</label>
                        <textarea class="form-control" id="system_prompt" rows="3"><?php echo htmlspecialchars($config['default_prompts']['article'] ?? ''); ?></textarea>
                    </div>
                    
                    <div class="row mb-4">
                        <div class="col-md-6">
                            <label for="temperature" class="form-label">
                                Temperature 
                                <span class="text-muted">(<span id="temperatureValue">0.7</span>)</span>
                            </label>
                            <input type="range" class="form-range" id="temperature" 
                                   min="0" max="1" step="0.1" value="0.7"
                                   oninput="document.getElementById('temperatureValue').textContent = this.value">
                        </div>
                        <div class="col-md-6">
                            <label for="max_tokens" class="form-label">最大トークン数</label>
                            <input type="number" class="form-control" id="max_tokens" 
                                   value="2000" min="100"
                                   oninput="updateTokenEstimate()">
                            <small id="charEstimate" class="text-muted">約800文字</small>
                        </div>
                    </div>
                    
                    <!-- 情報収集設定 -->
                    <div class="card mb-3 bg-light">
                        <div class="card-body py-3">
                            <h5 class="card-title mb-2">
                                <i class="bi bi-search"></i> 情報収集設定
                            </h5>
                            
                            <div class="form-check mb-2">
                                <input class="form-check-input" type="checkbox" id="enable_perplexity">
                                <label class="form-check-label" for="enable_perplexity">
                                    Perplexityで最新情報を収集する
                                </label>
                                <small class="text-muted d-block">
                                    記事のキーワードに基づいて、インターネットから最新情報を取得し、記事に反映します。
                                </small>
                            </div>
                        </div>
                    </div>
                    
                    <!-- 画像生成設定 -->
                    <div class="form-check mb-3">
                        <input class="form-check-input" type="checkbox" id="enable_images" onchange="toggleImageSettings()">
                        <label class="form-check-label" for="enable_images">
                            画像を自動生成する
                        </label>
                    </div>
                    
                    <div id="imageSettings" class="mb-4" style="display: none;">
                        <div class="row">
                            <div class="col-md-4">
                                <label for="image_model" class="form-label small">画像モデル</label>
                                <select class="form-select form-select-sm" id="image_model" onchange="updateImageSizeOptions()">
                                    <?php foreach ($config['image_models'] as $key => $model): ?>
                                        <option value="<?php echo htmlspecialchars($key); ?>">
                                            <?php echo htmlspecialchars($model['name']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-4">
                                <label for="image_style" class="form-label small">スタイル</label>
                                <select class="form-select form-select-sm" id="image_style">
                                    <?php foreach ($config['image_styles'] as $key => $style): ?>
                                        <option value="<?php echo htmlspecialchars($key); ?>">
                                            <?php echo htmlspecialchars($style); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-4">
                                <label for="num_images" class="form-label small">枚数</label>
                                <input type="number" class="form-control form-control-sm" id="num_images" 
                                       value="3" min="1" max="10" onchange="updateImagePrompts()">
                            </div>
                        </div>
                        
                        <div class="row mt-2">
                            <div class="col-md-6">
                                <label for="image_size" class="form-label small">画像サイズ</label>
                                <select class="form-select form-select-sm" id="image_size">
                                    <option value="1024x1024">正方形 (1024×1024)</option>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label for="image_quality" class="form-label small">画質設定 (DALL-E 3のみ)</label>
                                <select class="form-select form-select-sm" id="image_quality">
                                    <option value="standard">標準画質</option>
                                    <option value="hd">高画質 (HD)</option>
                                </select>
                            </div>
                        </div>
                        
                        <!-- 画像位置の設定（非表示） -->
                        <input type="hidden" id="image_position" value="after_heading">
                        
                        <div class="mt-2">
                            <label class="form-label small">画像プロンプト</label>
                            <div class="d-flex gap-3 mb-2">
                                <div class="form-check">
                                    <input class="form-check-input" type="radio" name="promptType" id="singlePrompt" value="single" checked onchange="togglePromptType()">
                                    <label class="form-check-label small" for="singlePrompt">共通</label>
                                </div>
                                <div class="form-check">
                                    <input class="form-check-input" type="radio" name="promptType" id="multiplePrompts" value="multiple" onchange="togglePromptType()">
                                    <label class="form-check-label small" for="multiplePrompts">個別</label>
                                </div>
                            </div>
                            
                            <div id="singlePromptArea">
                                <textarea class="form-control form-control-sm" id="image_prompt" rows="2"></textarea>
                            </div>
                            
                            <div id="multiplePromptsArea" style="display: none;">
                                <div id="imagePromptsList">
                                    <!-- 動的に生成 -->
                                </div>
                            </div>
                        </div>
                    </div>
                    
<?php if (isset($_GET['advanced']) && $_GET['advanced'] === 'mode2025'): ?>
                    <!-- アフィリエイトリンク設定 -->
                    <div class="card mb-3 bg-light">
                        <div class="card-body py-3">
                            <h5 class="card-title mb-2">
                                <i class="bi bi-link-45deg"></i> アフィリエイトリンク設定
                            </h5>
                            
                            <div class="form-check mb-2">
                                <input class="form-check-input" type="checkbox" id="enable_affiliate_links">
                                <label class="form-check-label" for="enable_affiliate_links">
                                    FANZA/DUGA検索リンクを挿入する
                                </label>
                                <small class="text-muted d-block">
                                    記事内にFANZAとDUGAの検索リンクをCTA形式で自動挿入します。
                                </small>
                            </div>
                            
                            <div class="form-check mb-2">
                                <input class="form-check-input" type="checkbox" id="enable_internal_links">
                                <label class="form-check-label" for="enable_internal_links">
                                    内部リンクを挿入する
                                </label>
                                <small class="text-muted d-block">
                                    関連する過去記事への魅力的な誘導リンクを自動挿入します。
                                </small>
                            </div>
                        </div>
                    </div>
<?php endif; ?>
                    
                    <!-- ユーザープロンプト -->
                    <div class="mb-4">
                        <label for="user_prompt" class="form-label">記事の内容（プロンプト）</label>
                        <textarea class="form-control" id="user_prompt" rows="5" required
                                  placeholder="生成したい記事の内容を詳しく説明してください。&#10;&#10;例：&#10;AIの最新トレンドについて、以下の点を含めて解説してください：&#10;- 生成AI（GPT-4、Claude、Gemini）の進化&#10;- 画像生成技術の発展&#10;- ビジネスへの応用事例&#10;- 今後の展望"></textarea>
                    </div>
                    
                    
                    <div class="mb-3">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="save_history" checked>
                            <label class="form-check-label" for="save_history">
                                履歴に保存
                            </label>
                        </div>
                    </div>
                    
                    <div class="d-flex gap-2 align-items-center">
                        <button type="submit" class="btn btn-primary btn-lg" id="generateBtn">
                            <i class="bi bi-magic"></i> 記事を生成
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <div class="col-md-4">
        <!-- 使用のヒント（上部に移動） -->
        <div class="card mb-3 border-primary" style="background-color: #ffffff;">
            <div class="card-header bg-primary text-white">
                <h5 class="mb-0"><i class="bi bi-lightbulb"></i> 使用のヒント</h5>
            </div>
            <div class="card-body" style="background-color: #ffffff; color: #212529;">
                <h6 style="color: #0d6efd; font-weight: 600;">効果的なプロンプトの書き方</h6>
                <ul class="small" style="color: #212529;">
                    <li>具体的な要求を明確に記載</li>
                    <li>希望する記事の構成を指定</li>
                    <li>ターゲット読者を明示</li>
                    <li>文体や雰囲気を指定</li>
                </ul>
                
                <h6 class="mt-3" style="color: #0d6efd; font-weight: 600;">Temperature設定</h6>
                <ul class="small" style="color: #212529;">
                    <li><strong>0.0-0.3:</strong> 事実重視、正確性重視</li>
                    <li><strong>0.4-0.7:</strong> バランス型（推奨）</li>
                    <li><strong>0.8-1.0:</strong> 創造的、独創的</li>
                </ul>
                
                <h6 class="mt-3" style="color: #0d6efd; font-weight: 600;">画像生成のコツ</h6>
                <ul class="small" style="color: #212529;">
                    <li>記事の内容に合った画像スタイルを選択</li>
                    <li>見出しごとに異なる画像を生成可能</li>
                    <li>プロンプトで具体的な要素を指定</li>
                </ul>
            </div>
        </div>
        
        <!-- モデル情報 -->
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-info-circle"></i> モデル情報</h5>
            </div>
            <div class="card-body" id="modelInfo">
                <p class="text-muted">モデルを選択してください</p>
            </div>
        </div>
    </div>
</div>

<!-- 結果表示エリア -->
<div class="modal fade" id="resultModal" tabindex="-1">
    <div class="modal-dialog modal-xl">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">生成結果</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <ul class="nav nav-tabs mb-3" id="resultTabs" role="tablist">
                    <li class="nav-item" role="presentation">
                        <button class="nav-link active" id="visual-tab" data-bs-toggle="tab" data-bs-target="#visual" type="button">
                            ビジュアル
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="html-tab" data-bs-toggle="tab" data-bs-target="#html" type="button">
                            HTML
                        </button>
                    </li>
                    <li class="nav-item" role="presentation">
                        <button class="nav-link" id="markdown-tab" data-bs-toggle="tab" data-bs-target="#markdown" type="button">
                            Markdown
                        </button>
                    </li>
                </ul>
                
                <div class="tab-content" id="resultTabContent">
                    <div class="tab-pane fade show active" id="visual" role="tabpanel">
                        <div id="resultContent"></div>
                    </div>
                    <div class="tab-pane fade" id="html" role="tabpanel">
                        <pre class="bg-light p-3 rounded"><code id="htmlContent"></code></pre>
                    </div>
                    <div class="tab-pane fade" id="markdown" role="tabpanel">
                        <pre class="bg-light p-3 rounded"><code id="markdownContent"></code></pre>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" onclick="copyToClipboard()">
                    <i class="bi bi-clipboard"></i> コピー
                </button>
                <button type="button" class="btn btn-info" onclick="saveAllImages()">
                    <i class="bi bi-image"></i> 画像を保存
                </button>
                <button type="button" class="btn btn-primary" onclick="downloadArticle()">
                    <i class="bi bi-download"></i> ダウンロード
                </button>
                <button type="button" class="btn btn-success" onclick="saveAsPreset()">
                    <i class="bi bi-bookmark"></i> プリセットとして保存
                </button>
            </div>
        </div>
    </div>
</div>

<script>
// グローバル変数（重複を防ぐためwindowオブジェクトを使用）
if (typeof window.currentArticle === 'undefined') {
    window.currentArticle = null;
}
if (typeof window.currentImages === 'undefined') {
    window.currentImages = [];
}
let currentMarkdown = '';
let originalContent = ''; // 元のHTMLコンテンツを保存

// プリセットデータ
let presets = <?php echo json_encode($presets); ?>;

// プリセット読み込み関数
function loadPreset() {
    const presetKey = document.getElementById('preset').value;
    if (!presetKey) return;
    
    const preset = presets[presetKey];
    if (!preset) return;
    
    // フォームに値を設定
    if (preset.model) {
        document.getElementById('model').value = preset.model;
        updateModelInfo(preset.model);
    }
    
    if (preset.temperature !== undefined) {
        document.getElementById('temperature').value = preset.temperature;
        document.getElementById('temperatureValue').textContent = preset.temperature;
    }
    
    if (preset.max_tokens) {
        document.getElementById('max_tokens').value = preset.max_tokens;
        updateTokenEstimate();
    }
    
    if (preset.system_prompt) {
        document.getElementById('system_prompt').value = preset.system_prompt;
    }
    
    // 画像設定
    if (preset.enable_image !== undefined) {
        document.getElementById('enable_images').checked = preset.enable_image;
        toggleImageSettings();
    }
    
    if (preset.image_model) {
        document.getElementById('image_model').value = preset.image_model;
        updateImageSizeOptions();
    }
    
    if (preset.image_style) {
        document.getElementById('image_style').value = preset.image_style;
    }
    
    if (preset.image_prompt) {
        document.getElementById('image_prompt').value = preset.image_prompt;
    }
    
    // Perplexity設定の読み込み
    if (preset.enable_perplexity !== undefined) {
        document.getElementById('enable_perplexity').checked = preset.enable_perplexity;
    }
    
    showToast('プリセット「' + preset.name + '」を読み込みました', 'success');
}

// フォーム送信処理
document.addEventListener('DOMContentLoaded', function() {
    console.log('DOMContentLoaded event fired');
    const form = document.getElementById('generateForm');
    
    if (!form) {
        console.error('generateForm not found!');
        return;
    }
    
    console.log('Form found, adding event listener');
    
    form.addEventListener('submit', async (e) => {
        e.preventDefault();
        console.log('Form submitted!');
        
        // Ctrl/Cmdキーが押されている場合は新しいタブで開く
        if (e.ctrlKey || e.metaKey) {
            generateInNewTab();
            return;
        }
        
        const btn = document.getElementById('generateBtn');
        if (!btn) {
            console.error('generateBtn not found!');
            return;
        }
        const originalText = btn.innerHTML;
        btn.disabled = true;
        btn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>生成中...';
        console.log('Starting generation...');
        
        try {
            // 画像プロンプトの収集
            let imagePrompts = [];
            const isMultiplePrompts = document.getElementById('multiplePrompts').checked;
            
            if (isMultiplePrompts) {
                const promptElements = document.querySelectorAll('.image-prompt-item');
                promptElements.forEach(el => {
                    imagePrompts.push(el.value);
                });
            } else {
                // 単一プロンプトの場合は配列として送信
                const singlePrompt = document.getElementById('image_prompt')?.value || '';
                const numImages = parseInt(document.getElementById('num_images')?.value || 3);
                for (let i = 0; i < numImages; i++) {
                    imagePrompts.push(singlePrompt);
                }
            }
            
            // カテゴリーの取得（カスタムカテゴリーを優先）
            const customCategory = document.getElementById('custom_category')?.value?.trim();
            const selectedCategory = document.getElementById('category')?.value;
            const category = customCategory || selectedCategory || 'その他';
            
            const formData = {
                title: document.getElementById('title').value,
                keywords: document.getElementById('keywords')?.value || '',
                category: category,
                model: document.getElementById('model').value,
                output_format: document.getElementById('output_format')?.value || 'visual',
                system_prompt: document.getElementById('system_prompt')?.value || '',
                user_prompt: document.getElementById('user_prompt').value,
                temperature: parseFloat(document.getElementById('temperature')?.value || 0.7),
                max_tokens: parseInt(document.getElementById('max_tokens')?.value || 2000),
                save_history: document.getElementById('save_history')?.checked ?? true,
                enable_perplexity: document.getElementById('enable_perplexity')?.checked ?? false,
                enable_images: document.getElementById('enable_images')?.checked ?? false,
                enable_affiliate_links: document.getElementById('enable_affiliate_links')?.checked ?? false,
                enable_internal_links: document.getElementById('enable_internal_links')?.checked ?? false,
                image_model: document.getElementById('image_model')?.value || 'dall-e-2',
                image_style: document.getElementById('image_style')?.value || 'default',
                image_prompts: imagePrompts,
                num_images: parseInt(document.getElementById('num_images')?.value || 3),
                image_position: document.getElementById('image_position')?.value || 'after_heading',
                image_size: document.getElementById('image_size')?.value || '1024x1024',
                image_quality: document.getElementById('image_quality')?.value || 'standard'
            };
            
            console.log('Form data:', formData);
            
            const response = await fetch('../api/service.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json; charset=utf-8',
                    'Accept': 'application/json; charset=utf-8'
                },
                credentials: 'same-origin', // セッションクッキーを送信
                body: JSON.stringify({
                    action: 'generate',
                    ...formData
                })
            });
            
            const responseText = await response.text();
            
            let result;
            try {
                result = JSON.parse(responseText);
            } catch (e) {
                console.error('JSON Parse Error:', e);
                console.error('Response Text:', responseText);
                
                if (responseText.includes('<!DOCTYPE') || responseText.includes('<html')) {
                    throw new Error('サーバーエラー: HTMLレスポンスが返されました。ログインが必要な可能性があります。');
                } else if (responseText === '') {
                    throw new Error('サーバーエラー: 空のレスポンスが返されました。');
                } else {
                    throw new Error('サーバーエラー: 不正なJSONレスポンス - ' + responseText.substring(0, 100));
                }
            }
            
            if (result.success) {
                currentArticle = result.content;
                currentImages = result.images || [];
                originalContent = result.content;
                
                // モデル設定をsessionStorageに保存（SNS投稿生成用）
                const modelSettings = {
                    model: formData.model,
                    temperature: formData.temperature || 0.7,
                    top_p: formData.top_p || 0.9
                };
                sessionStorage.setItem('lastModelSettings', JSON.stringify(modelSettings));
                console.log('Saved model settings to sessionStorage:', modelSettings);
                
                showResult(result, formData);
            } else {
                alert('エラー: ' + result.error);
            }
            
        } catch (error) {
            alert('通信エラーが発生しました: ' + error.message);
        } finally {
            btn.disabled = false;
            btn.innerHTML = originalText;
        }
    });
    
    // モデル変更時の処理
    document.getElementById('model').addEventListener('change', function() {
        updateModelInfo(this.value);
    });
    
    // 初期モデル情報表示
    updateModelInfo(document.getElementById('model').value);
});

// 削除: generateInNewTab関数（新しいタブ機能は使用しない）

function updateModelInfo(modelKey) {
    const models = <?php echo json_encode($config['models']); ?>;
    let modelInfo = null;
    
    for (const provider in models) {
        if (models[provider][modelKey]) {
            modelInfo = models[provider][modelKey];
            break;
        }
    }
    
    if (modelInfo) {
        document.getElementById('modelInfo').innerHTML = `
            <h6>${modelInfo.name}</h6>
            <p class="small mb-2">${modelInfo.description}</p>
            <p class="small mb-0">最大トークン: ${modelInfo.max_tokens.toLocaleString()}</p>
        `;
    }
}

// トークン数から推定文字数を更新
function updateTokenEstimate() {
    const tokens = parseInt(document.getElementById('max_tokens').value) || 0;
    const estimatedChars = Math.round(tokens / 2.5); // 日本語の場合、1文字≈2.5トークン
    document.getElementById('charEstimate').textContent = `約${estimatedChars.toLocaleString()}文字`;
}

// 画像モデルに応じてサイズオプションを更新
function updateImageSizeOptions() {
    const model = document.getElementById('image_model').value;
    const sizeSelect = document.getElementById('image_size');
    const qualitySelect = document.getElementById('image_quality');
    const imageSizes = <?php echo json_encode($config['image_sizes'] ?? []); ?>;
    
    // サイズオプションをクリア
    sizeSelect.innerHTML = '';
    
    // モデルに応じたサイズを設定
    let sizes = imageSizes['default'] || {'1024x1024': '正方形 (1024×1024)'};
    
    if (model === 'dall-e-3' && imageSizes['dall-e-3']) {
        sizes = imageSizes['dall-e-3'];
        qualitySelect.disabled = false;
    } else if ((model.startsWith('venice-') || model.startsWith('flux-')) && imageSizes['venice']) {
        sizes = imageSizes['venice'];
        qualitySelect.disabled = true;
        qualitySelect.value = 'standard';
    } else if (model === 'gemini-image' && imageSizes['gemini']) {
        sizes = imageSizes['gemini'];
        qualitySelect.disabled = true;
        qualitySelect.value = 'standard';
    }
    
    // サイズオプションを追加
    for (const [value, label] of Object.entries(sizes)) {
        const option = document.createElement('option');
        option.value = value;
        option.textContent = label;
        sizeSelect.appendChild(option);
    }
}

function toggleImageSettings() {
    const enabled = document.getElementById('enable_images').checked;
    document.getElementById('imageSettings').style.display = enabled ? 'block' : 'none';
    if (enabled) {
        updateImagePrompts();
        updateImageSizeOptions();
    }
}

// プロンプトタイプの切り替え
function togglePromptType() {
    const isMultiple = document.getElementById('multiplePrompts').checked;
    document.getElementById('singlePromptArea').style.display = isMultiple ? 'none' : 'block';
    document.getElementById('multiplePromptsArea').style.display = isMultiple ? 'block' : 'none';
    
    if (isMultiple) {
        updateImagePrompts();
    }
}

// 画像プロンプトフィールドの更新
function updateImagePrompts() {
    const numImages = parseInt(document.getElementById('num_images').value);
    const container = document.getElementById('imagePromptsList');
    container.innerHTML = '';
    
    for (let i = 1; i <= numImages; i++) {
        const div = document.createElement('div');
        div.className = 'mb-2';
        div.innerHTML = `
            <label class="form-label">画像 ${i} のプロンプト</label>
            <textarea class="form-control image-prompt-item" rows="2" placeholder="画像 ${i} の生成プロンプト"></textarea>
        `;
        container.appendChild(div);
    }
}

function showResult(result, formData) {
    let content = result.content;
    
    // 画像の挿入
    if (result.images && result.images.length > 0) {
        result.images.forEach(img => {
            const imgHtml = `
                <figure class="text-center my-4">
                    <img src="${img.url}" alt="${img.alt}" class="img-fluid rounded shadow">
                    <figcaption class="text-muted small mt-2">${img.alt}</figcaption>
                </figure>
            `;
            
            // HTMLの見出しタグの後に画像を挿入
            const escapedSection = img.section.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
            
            // h1, h2, h3タグを探す
            const headingRegex = new RegExp(`(<h[1-3][^>]*>[^<]*${escapedSection}[^<]*</h[1-3]>)`, 'gi');
            if (content.match(headingRegex)) {
                content = content.replace(headingRegex, `$1\n${imgHtml}`);
            } else {
                // 見出しが見つからない場合は、テキストを直接探す
                const textRegex = new RegExp(`(${escapedSection})`, 'gi');
                const matches = content.match(textRegex);
                if (matches && matches.length > 0) {
                    // 最初のマッチの後に画像を挿入
                    content = content.replace(matches[0], `${matches[0]}\n${imgHtml}`);
                }
            }
        });
    }
    
    // ビジュアルタブに表示
    document.getElementById('resultContent').innerHTML = content;
    
    // HTMLタブに表示
    const htmlTextNode = document.createTextNode(content);
    const htmlContentEl = document.getElementById('htmlContent');
    htmlContentEl.textContent = '';
    htmlContentEl.appendChild(htmlTextNode);
    
    // Markdownタブに変換して表示
    currentMarkdown = convertToMarkdown(content);
    const mdTextNode = document.createTextNode(currentMarkdown);
    const mdContentEl = document.getElementById('markdownContent');
    mdContentEl.textContent = '';
    mdContentEl.appendChild(mdTextNode);
    
    // メタデータの表示
    if (result.metadata) {
        const metaHtml = `
            <div class="alert alert-info mt-4">
                <small>
                    <strong>モデル:</strong> ${result.metadata.model} | 
                    <strong>作成日時:</strong> ${result.metadata.timestamp} | 
                    <strong>キーワード:</strong> ${formData.keywords || '未設定'}
                </small>
            </div>
        `;
        document.getElementById('resultContent').innerHTML += metaHtml;
    }
    
    new bootstrap.Modal(document.getElementById('resultModal')).show();
}

// その他のユーティリティ関数
async function copyToClipboard() {
    let contentToCopy = '';
    
    // アクティブなタブを確認
    const activeTab = document.querySelector('#resultTabs .nav-link.active').id;
    
    switch (activeTab) {
        case 'visual-tab':
            contentToCopy = originalContent;
            break;
        case 'html-tab':
            contentToCopy = document.getElementById('htmlContent').textContent;
            break;
        case 'markdown-tab':
            contentToCopy = document.getElementById('markdownContent').textContent;
            break;
    }
    
    try {
        await navigator.clipboard.writeText(contentToCopy);
        showToast('クリップボードにコピーしました', 'success');
    } catch (error) {
        showToast('コピーに失敗しました: ' + error.message, 'danger');
    }
}

function downloadArticle() {
    if (!currentArticle) return;
    
    const format = prompt('ダウンロード形式を選択してください (html/markdown/text):', 'html');
    if (!format) return;
    
    let content = currentArticle;
    let filename = 'article';
    let mimeType = 'text/plain';
    
    switch (format.toLowerCase()) {
        case 'html':
            content = wrapInHTML(content, currentImages);
            filename += '.html';
            mimeType = 'text/html';
            break;
        case 'markdown':
            content = currentMarkdown;
            filename += '.md';
            mimeType = 'text/markdown';
            break;
        case 'text':
            content = stripTags(content);
            filename += '.txt';
            break;
    }
    
    const blob = new Blob([content], { type: mimeType });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    a.click();
    URL.revokeObjectURL(url);
    
    showToast('ダウンロードを開始しました', 'success');
}

function saveAsPreset() {
    const name = prompt('プリセット名を入力してください:');
    if (!name) return;
    
    const presetData = {
        name: name,
        model: document.getElementById('model').value,
        temperature: parseFloat(document.getElementById('temperature').value),
        max_tokens: parseInt(document.getElementById('max_tokens').value),
        system_prompt: document.getElementById('system_prompt').value,
        enable_image: document.getElementById('enable_images').checked,
        image_model: document.getElementById('image_model').value,
        image_style: document.getElementById('image_style').value,
        image_prompt: document.getElementById('image_prompt').value,
        enable_perplexity: document.getElementById('enable_perplexity').checked
    };
    
    const formData = new FormData();
    formData.append('ajax', '1');
    formData.append('action', 'save');
    formData.append('name', presetData.name);
    formData.append('model', presetData.model);
    formData.append('temperature', presetData.temperature);
    formData.append('max_tokens', presetData.max_tokens);
    formData.append('system_prompt', presetData.system_prompt);
    if (presetData.enable_image) {
        formData.append('enable_image', '1');
    }
    formData.append('image_model', presetData.image_model);
    formData.append('image_style', presetData.image_style);
    formData.append('image_prompt', presetData.image_prompt);
    if (presetData.enable_perplexity) {
        formData.append('enable_perplexity', '1');
    }
    
    fetch('../api/preset-service.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(result => {
        if (result.success) {
            showToast('プリセットを保存しました', 'success');
            setTimeout(() => location.reload(), 1000);
        } else {
            showToast('保存に失敗しました: ' + result.error, 'danger');
        }
    })
    .catch(error => {
        showToast('エラーが発生しました: ' + error.message, 'danger');
    });
}

function showToast(message, type = 'info') {
    const toastHtml = `
        <div class="toast align-items-center text-white bg-${type} border-0" role="alert">
            <div class="d-flex">
                <div class="toast-body">${message}</div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
            </div>
        </div>
    `;
    
    let container = document.querySelector('.toast-container');
    if (!container) {
        container = document.createElement('div');
        container.className = 'toast-container position-fixed top-0 end-0 p-3';
        container.style.zIndex = '9999';
        document.body.appendChild(container);
    }
    
    container.insertAdjacentHTML('beforeend', toastHtml);
    const toastEl = container.lastElementChild;
    const toast = new bootstrap.Toast(toastEl);
    toast.show();
    
    toastEl.addEventListener('hidden.bs.toast', () => {
        toastEl.remove();
    });
}

function wrapInHTML(content, images) {
    content = content.replace(/^﻿/, '');
    
    return `<!DOCTYPE html>
<html lang="ja">
<head>
    <meta charset="UTF-8">
    <meta http-equiv="Content-Type" content="text/html; charset=UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Generated Article</title>
    <style>
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            line-height: 1.6;
            color: #333;
            max-width: 800px;
            margin: 0 auto;
            padding: 20px;
        }
        h1, h2, h3 { color: #2c3e50; }
        img { max-width: 100%; height: auto; }
        figure { margin: 2em 0; text-align: center; }
        figcaption { margin-top: 0.5em; color: #666; font-size: 0.9em; }
    </style>
</head>
<body>
    ${content}
</body>
</html>`;
}

function convertToMarkdown(html) {
    let markdown = html;
    
    // 画像の変換を最初に行う
    markdown = markdown.replace(/<figure[^>]*>.*?<img[^>]*src="([^"]+)"[^>]*alt="([^"]*)"[^>]*>.*?<figcaption[^>]*>([^<]*)<\/figcaption>.*?<\/figure>/gi, '![$2]($1)\n*$3*\n\n');
    markdown = markdown.replace(/<img[^>]*src="([^"]+)"[^>]*alt="([^"]*)"[^>]*>/gi, '![$2]($1)\n\n');
    
    // HTMLタグをMarkdownに変換
    markdown = markdown.replace(/<h1[^>]*>(.+?)<\/h1>/gi, '# $1\n\n');
    markdown = markdown.replace(/<h2[^>]*>(.+?)<\/h2>/gi, '## $1\n\n');
    markdown = markdown.replace(/<h3[^>]*>(.+?)<\/h3>/gi, '### $1\n\n');
    markdown = markdown.replace(/<strong[^>]*>(.+?)<\/strong>/gi, '**$1**');
    markdown = markdown.replace(/<em[^>]*>(.+?)<\/em>/gi, '*$1*');
    markdown = markdown.replace(/<p[^>]*>(.+?)<\/p>/gi, '$1\n\n');
    markdown = markdown.replace(/<br\s*\/?>/gi, '\n');
    
    // リストの変換
    markdown = markdown.replace(/<ul[^>]*>/gi, '');
    markdown = markdown.replace(/<\/ul>/gi, '\n');
    markdown = markdown.replace(/<li[^>]*>(.+?)<\/li>/gi, '- $1\n');
    
    // 残りのHTMLタグを削除
    markdown = markdown.replace(/<[^>]+>/g, '');
    
    // 複数の改行を2つまでに制限
    markdown = markdown.replace(/\n{3,}/g, '\n\n');
    
    return markdown.trim();
}

function stripTags(html) {
    const tmp = document.createElement('div');
    tmp.innerHTML = html;
    return tmp.textContent || tmp.innerText || '';
}

function saveAllImages() {
    if (!currentImages || currentImages.length === 0) {
        showToast('保存する画像がありません', 'warning');
        return;
    }
    
    const articleTitle = document.getElementById('title').value || 'article';
    window.downloadAllImages(currentImages, articleTitle);
}

// カテゴリー選択の処理
function toggleCustomCategory() {
    const categorySelect = document.getElementById('category');
    const selectMode = document.getElementById('select_mode');
    const customMode = document.getElementById('custom_mode');
    const customInput = document.getElementById('custom_category');
    const hint = document.getElementById('category_hint');
    
    if (categorySelect.value === '__new__') {
        // 新規カテゴリーモードに切り替え
        selectMode.style.display = 'none';
        customMode.style.display = 'block';
        customInput.value = '';
        customInput.focus();
        hint.innerHTML = '<i class="bi bi-info-circle"></i> カテゴリー名を入力してください。キャンセルで選択モードに戻ります。';
    }
}

function cancelCustomCategory() {
    const selectMode = document.getElementById('select_mode');
    const customMode = document.getElementById('custom_mode');
    const categorySelect = document.getElementById('category');
    const customInput = document.getElementById('custom_category');
    const hint = document.getElementById('category_hint');
    
    // 選択モードに戻る
    selectMode.style.display = 'block';
    customMode.style.display = 'none';
    categorySelect.value = '';
    customInput.value = '';
    hint.innerHTML = '<i class="bi bi-info-circle"></i> 既存のカテゴリーを選択するか、「新規カテゴリーを追加」を選んで作成できます';
}

// 初期化処理
document.addEventListener('DOMContentLoaded', function() {
    updateImageSizeOptions();
    
    // カテゴリー入力でEnterキーを無効化
    document.getElementById('custom_category').addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            e.preventDefault();
        }
    });
});
</script>

<style>
/* タブのスタイル */
.nav-tabs .nav-link {
    color: #495057;
}

.nav-tabs .nav-link.active {
    color: #667eea;
    font-weight: 600;
}

/* プレタグのスタイル */
pre {
    max-height: 600px;
    overflow-y: auto;
}

pre code {
    font-size: 0.9rem;
    white-space: pre-wrap;
    word-wrap: break-word;
}
</style>