<?php
// プリセットの読み込み
$presetsFile = __DIR__ . '/../../data/presets.json';
$presets = [];
if (file_exists($presetsFile)) {
    $presets = json_decode(file_get_contents($presetsFile), true) ?: [];
}

// 新しいタブからのパラメータを受け取る
$fromNewTab = isset($_GET['new_tab']) && $_GET['new_tab'] == '1';
$autoTitle = $_GET['title'] ?? '';
$autoKeywords = $_GET['keywords'] ?? '';
$autoModel = $_GET['model'] ?? '';
$autoPrompt = $_GET['prompt'] ?? '';
?>

<div class="page-header">
    <h1><i class="bi bi-file-text"></i> 記事生成</h1>
    <p class="text-muted">AIを使用して高品質な記事を生成します</p>
</div>

<?php if ($fromNewTab): ?>
<div class="alert alert-info">
    <i class="bi bi-info-circle"></i> 新しいタブで記事を生成中です...
</div>
<?php endif; ?>

<div class="row">
    <div class="col-md-8">
        <div class="card">
            <div class="card-body">
                <form id="generateForm" method="POST">
                    <!-- 基本設定 -->
                    <div class="mb-4">
                        <label for="title" class="form-label">記事タイトル</label>
                        <input type="text" class="form-control" id="title" name="title"
                               value="<?php echo htmlspecialchars($autoTitle); ?>"
                               placeholder="例：AIの最新トレンド2025" required>
                    </div>
                    
                    <div class="mb-4">
                        <label for="keywords" class="form-label">キーワード</label>
                        <input type="text" class="form-control" id="keywords" name="keywords"
                               value="<?php echo htmlspecialchars($autoKeywords); ?>"
                               placeholder="カンマ区切りで入力（例：AI, 人工知能, 機械学習）">
                    </div>
                    
                    <div class="row mb-4">
                        <div class="col-md-6">
                            <label for="model" class="form-label">AIモデル</label>
                            <select class="form-control" id="model" name="model">
                                <?php foreach ($config['models'] as $provider => $models): ?>
                                    <optgroup label="<?php echo ucfirst($provider); ?>">
                                        <?php foreach ($models as $key => $model): ?>
                                            <option value="<?php echo htmlspecialchars($key); ?>" 
                                                    <?php echo ($autoModel && $key === $autoModel) || (!$autoModel && $key === $config['default_model']) ? 'selected' : ''; ?>>
                                                <?php echo htmlspecialchars($model['name']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </optgroup>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-6">
                            <label for="output_format" class="form-label">出力形式</label>
                            <select class="form-control" id="output_format" name="output_format">
                                <option value="visual">ビジュアルエディタ</option>
                                <option value="html">HTML</option>
                                <option value="markdown">Markdown</option>
                                <option value="text">プレーンテキスト</option>
                            </select>
                        </div>
                    </div>
                    
                    <!-- ユーザープロンプト -->
                    <div class="mb-4">
                        <label for="user_prompt" class="form-label">記事の内容（プロンプト）</label>
                        <textarea class="form-control" id="user_prompt" name="user_prompt" rows="5" required
                                  placeholder="生成したい記事の内容を詳しく説明してください。"><?php echo htmlspecialchars($autoPrompt); ?></textarea>
                    </div>
                    
                    <div class="mb-3">
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" id="save_history" name="save_history" checked>
                            <label class="form-check-label" for="save_history">
                                履歴に保存
                            </label>
                        </div>
                    </div>
                    
                    <div class="d-flex gap-2 align-items-center">
                        <button type="submit" class="btn btn-primary btn-lg" id="generateBtn">
                            <i class="bi bi-magic"></i> 記事を生成
                        </button>
                        <button type="button" class="btn btn-outline-primary btn-lg" 
                                onclick="generateInNewTab()"
                                title="新しいタブで生成（複数同時生成可能）">
                            <i class="bi bi-box-arrow-up-right"></i> 新しいタブで生成
                        </button>
                    </div>
                </form>
            </div>
        </div>
        
        <!-- 生成状況表示エリア -->
        <div id="statusArea" class="mt-4" style="display: none;">
            <div class="card">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0"><i class="bi bi-hourglass-split"></i> 生成状況</h5>
                </div>
                <div class="card-body">
                    <div class="progress mb-3" style="height: 25px;">
                        <div id="progressBar" class="progress-bar progress-bar-striped progress-bar-animated" 
                             role="progressbar" style="width: 0%">0%</div>
                    </div>
                    <p id="statusText" class="text-center">記事を生成しています...</p>
                </div>
            </div>
        </div>
        
        <!-- 結果表示エリア -->
        <div id="resultArea" class="mt-4" style="display: none;">
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">生成結果</h5>
                </div>
                <div class="card-body" id="resultContent">
                    <!-- 結果がここに表示される -->
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// グローバル変数
let currentArticle = null;
let progressTimer = null;

// フォーム送信処理
document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('generateForm');
    
    // 新しいタブからの自動実行
    <?php if ($fromNewTab): ?>
    setTimeout(() => {
        console.log('Auto-generating from new tab...');
        document.getElementById('generateBtn').click();
    }, 500);
    <?php endif; ?>
    
    form.addEventListener('submit', async (e) => {
        e.preventDefault();
        
        const btn = document.getElementById('generateBtn');
        const originalText = btn.innerHTML;
        btn.disabled = true;
        btn.innerHTML = '<span class="spinner-border spinner-border-sm me-2"></span>生成中...';
        
        // 進捗表示を開始
        showProgressBar();
        
        try {
            const formData = {
                action: 'generate',
                title: document.getElementById('title').value,
                keywords: document.getElementById('keywords').value,
                model: document.getElementById('model').value,
                output_format: document.getElementById('output_format').value,
                user_prompt: document.getElementById('user_prompt').value,
                temperature: 0.7,
                max_tokens: 2000,
                save_history: document.getElementById('save_history').checked,
                enable_perplexity: false,
                enable_images: false
            };
            
            const response = await fetch('../api/service.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json; charset=utf-8'
                },
                credentials: 'same-origin',
                body: JSON.stringify(formData)
            });
            
            const responseText = await response.text();
            
            let result;
            try {
                result = JSON.parse(responseText);
            } catch (e) {
                console.error('JSON Parse Error:', e);
                throw new Error('サーバーエラー: 不正なレスポンス');
            }
            
            if (result.success) {
                currentArticle = result.content;
                hideProgressBar();
                document.getElementById('resultArea').style.display = 'block';
                document.getElementById('resultContent').innerHTML = result.content;
                
                // タイトルを更新
                document.title = document.getElementById('title').value + ' - 生成完了';
                
                // 成功メッセージ
                showToast('記事生成が完了しました！', 'success');
            } else {
                hideProgressBar();
                alert('エラー: ' + result.error);
            }
            
        } catch (error) {
            hideProgressBar();
            alert('通信エラーが発生しました: ' + error.message);
        } finally {
            btn.disabled = false;
            btn.innerHTML = originalText;
        }
    });
});

// 新しいタブで生成する関数
function generateInNewTab() {
    const form = document.getElementById('generateForm');
    
    // フォームのバリデーションチェック
    if (!form.checkValidity()) {
        form.reportValidity();
        return;
    }
    
    // URLパラメータを作成
    const params = new URLSearchParams({
        'page': 'generate',
        'new_tab': '1',
        'title': document.getElementById('title').value,
        'keywords': document.getElementById('keywords').value,
        'model': document.getElementById('model').value,
        'prompt': document.getElementById('user_prompt').value
    });
    
    // 新しいタブで開く
    window.open('?' + params.toString(), '_blank');
}

// 進捗バー表示
function showProgressBar() {
    document.getElementById('statusArea').style.display = 'block';
    let progress = 0;
    
    progressTimer = setInterval(() => {
        progress += Math.random() * 15;
        if (progress > 90) progress = 90;
        
        document.getElementById('progressBar').style.width = progress + '%';
        document.getElementById('progressBar').textContent = Math.round(progress) + '%';
        
        // ステータステキストを更新
        const messages = [
            '記事を生成しています...',
            'AIが内容を検討しています...',
            '文章を構成しています...',
            'もう少しで完成します...'
        ];
        const messageIndex = Math.floor(progress / 25);
        document.getElementById('statusText').textContent = messages[messageIndex] || messages[0];
    }, 1000);
}

// 進捗バー非表示
function hideProgressBar() {
    if (progressTimer) {
        clearInterval(progressTimer);
        progressTimer = null;
    }
    
    // 100%にしてから非表示
    document.getElementById('progressBar').style.width = '100%';
    document.getElementById('progressBar').textContent = '100%';
    document.getElementById('statusText').textContent = '完成しました！';
    
    setTimeout(() => {
        document.getElementById('statusArea').style.display = 'none';
    }, 1000);
}

// トースト表示関数
function showToast(message, type = 'info') {
    const toastHtml = `
        <div class="toast align-items-center text-white bg-${type} border-0" role="alert">
            <div class="d-flex">
                <div class="toast-body">${message}</div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast"></button>
            </div>
        </div>
    `;
    
    let container = document.querySelector('.toast-container');
    if (!container) {
        container = document.createElement('div');
        container.className = 'toast-container position-fixed top-0 end-0 p-3';
        container.style.zIndex = '9999';
        document.body.appendChild(container);
    }
    
    container.insertAdjacentHTML('beforeend', toastHtml);
    const toastEl = container.lastElementChild;
    const toast = new bootstrap.Toast(toastEl);
    toast.show();
    
    toastEl.addEventListener('hidden.bs.toast', () => {
        toastEl.remove();
    });
}
</script>