<?php
// カテゴリー管理ページ

// 記事データからカテゴリー一覧を取得
$historyFile = __DIR__ . '/../../data/history.json';
$articlesFile = __DIR__ . '/../../data/articles.json';
$history = [];
$articles = [];

if (file_exists($historyFile)) {
    $history = json_decode(file_get_contents($historyFile), true) ?: [];
}
if (file_exists($articlesFile)) {
    $articles = json_decode(file_get_contents($articlesFile), true) ?: [];
}

// 両方のファイルからカテゴリーごとの記事数をカウント
$categoryCount = [];

// historyからカウント
foreach ($history as $article) {
    $category = $article['category'] ?? 'その他';
    if (!isset($categoryCount[$category])) {
        $categoryCount[$category] = 0;
    }
    $categoryCount[$category]++;
}

// articlesからもカウント（重複を避けるため、IDをチェック）
$countedIds = array_column($history, 'id');
foreach ($articles as $article) {
    if (!in_array($article['id'], $countedIds)) {
        $category = $article['category'] ?? 'その他';
        if (!isset($categoryCount[$category])) {
            $categoryCount[$category] = 0;
        }
        $categoryCount[$category]++;
    }
}

// カテゴリーをソート
ksort($categoryCount);

// カテゴリー削除処理
if (isset($_POST['action']) && $_POST['action'] === 'delete_category') {
    $deleteCategory = $_POST['category'] ?? '';
    $newCategory = $_POST['new_category'] ?? 'その他';
    
    if ($deleteCategory && $deleteCategory !== $newCategory) {
        // 該当カテゴリーの記事を新しいカテゴリーに移動
        $updated = false;
        foreach ($history as &$article) {
            if (($article['category'] ?? 'その他') === $deleteCategory) {
                $article['category'] = $newCategory;
                $updated = true;
            }
        }
        
        if ($updated) {
            file_put_contents($historyFile, json_encode($history, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
            $_SESSION['category_message'] = "カテゴリー「{$deleteCategory}」を削除し、記事を「{$newCategory}」に移動しました。";
            $_SESSION['category_message_type'] = 'success';
        }
    }
    
    header('Location: ?page=categories');
    exit;
}

// カテゴリー名変更処理
if (isset($_POST['action']) && $_POST['action'] === 'rename_category') {
    $oldName = $_POST['old_name'] ?? '';
    $newName = $_POST['new_name'] ?? '';
    
    if ($oldName && $newName && $oldName !== $newName) {
        // 該当カテゴリーの記事を更新
        $updated = false;
        foreach ($history as &$article) {
            if (($article['category'] ?? 'その他') === $oldName) {
                $article['category'] = $newName;
                $updated = true;
            }
        }
        
        if ($updated) {
            file_put_contents($historyFile, json_encode($history, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
            $_SESSION['category_message'] = "カテゴリー名を「{$oldName}」から「{$newName}」に変更しました。";
            $_SESSION['category_message_type'] = 'success';
        }
    }
    
    header('Location: ?page=categories');
    exit;
}
?>

<div class="page-header">
    <h1><i class="bi bi-tags"></i> カテゴリー管理</h1>
    <p class="text-muted">記事のカテゴリーを管理します</p>
</div>

<?php if (isset($_SESSION['category_message'])): ?>
<div class="alert alert-<?php echo $_SESSION['category_message_type'] ?? 'info'; ?> alert-dismissible fade show" role="alert">
    <?php echo htmlspecialchars($_SESSION['category_message']); ?>
    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
</div>
<?php
unset($_SESSION['category_message']);
unset($_SESSION['category_message_type']);
?>
<?php endif; ?>

<div class="row">
    <div class="col-lg-8">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0">カテゴリー一覧</h5>
            </div>
            <div class="card-body">
                <?php if (empty($categoryCount)): ?>
                    <p class="text-center text-muted py-5">カテゴリーがありません。</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>カテゴリー名</th>
                                    <th style="width: 100px;">記事数</th>
                                    <th style="width: 150px;">操作</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($categoryCount as $category => $count): ?>
                                <tr>
                                    <td>
                                        <span class="fw-bold"><?php echo htmlspecialchars($category); ?></span>
                                    </td>
                                    <td>
                                        <span class="badge bg-info"><?php echo $count; ?> 記事</span>
                                    </td>
                                    <td>
                                        <button type="button" class="btn btn-sm btn-outline-primary" 
                                                onclick="showRenameModal('<?php echo htmlspecialchars($category, ENT_QUOTES); ?>')">
                                            <i class="bi bi-pencil"></i> 名前変更
                                        </button>
                                        <?php if ($category !== 'その他'): ?>
                                        <button type="button" class="btn btn-sm btn-outline-danger" 
                                                onclick="showDeleteModal('<?php echo htmlspecialchars($category, ENT_QUOTES); ?>', <?php echo $count; ?>)">
                                            <i class="bi bi-trash"></i>
                                        </button>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <div class="col-lg-4">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-info-circle"></i> カテゴリー情報</h5>
            </div>
            <div class="card-body">
                <h6>総カテゴリー数</h6>
                <p class="h4 text-primary"><?php echo count($categoryCount); ?></p>
                
                <hr>
                
                <h6>総記事数</h6>
                <p class="h4 text-success"><?php echo array_sum($categoryCount); ?></p>
                
                <hr>
                
                <h6>カテゴリー管理のヒント</h6>
                <ul class="small">
                    <li>「その他」カテゴリーは削除できません</li>
                    <li>カテゴリーを削除すると、該当記事は指定したカテゴリーに移動します</li>
                    <li>カテゴリー名は記事生成時に新規作成も可能です</li>
                </ul>
            </div>
        </div>
    </div>
</div>

<!-- 名前変更モーダル -->
<div class="modal fade" id="renameModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="">
                <input type="hidden" name="action" value="rename_category">
                <input type="hidden" name="old_name" id="renameOldName">
                
                <div class="modal-header">
                    <h5 class="modal-title">カテゴリー名変更</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="mb-3">
                        <label class="form-label">現在のカテゴリー名</label>
                        <input type="text" class="form-control" id="displayOldName" readonly>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">新しいカテゴリー名</label>
                        <input type="text" class="form-control" name="new_name" required>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">キャンセル</button>
                    <button type="submit" class="btn btn-primary">変更</button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- 削除確認モーダル -->
<div class="modal fade" id="deleteModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <form method="POST" action="">
                <input type="hidden" name="action" value="delete_category">
                <input type="hidden" name="category" id="deleteCategory">
                
                <div class="modal-header">
                    <h5 class="modal-title">カテゴリー削除確認</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-warning">
                        <i class="bi bi-exclamation-triangle"></i> 
                        カテゴリー「<span id="deleteCategoryName"></span>」を削除します。
                        <br>
                        このカテゴリーには <strong id="deleteArticleCount"></strong> 件の記事があります。
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">記事の移動先カテゴリー</label>
                        <select class="form-control" name="new_category" required>
                            <option value="その他">その他</option>
                            <?php foreach ($categoryCount as $category => $count): ?>
                                <?php if ($category !== 'その他'): ?>
                                <option value="<?php echo htmlspecialchars($category); ?>">
                                    <?php echo htmlspecialchars($category); ?>
                                </option>
                                <?php endif; ?>
                            <?php endforeach; ?>
                        </select>
                        <small class="form-text text-muted">削除するカテゴリーの記事はこのカテゴリーに移動されます</small>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">キャンセル</button>
                    <button type="submit" class="btn btn-danger">削除</button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
function showRenameModal(categoryName) {
    document.getElementById('renameOldName').value = categoryName;
    document.getElementById('displayOldName').value = categoryName;
    new bootstrap.Modal(document.getElementById('renameModal')).show();
}

function showDeleteModal(categoryName, articleCount) {
    document.getElementById('deleteCategory').value = categoryName;
    document.getElementById('deleteCategoryName').textContent = categoryName;
    document.getElementById('deleteArticleCount').textContent = articleCount;
    
    // 移動先から現在のカテゴリーを除外
    const selectElement = document.querySelector('#deleteModal select[name="new_category"]');
    Array.from(selectElement.options).forEach(option => {
        if (option.value === categoryName) {
            option.style.display = 'none';
            option.disabled = true;
        } else {
            option.style.display = '';
            option.disabled = false;
        }
    });
    
    new bootstrap.Modal(document.getElementById('deleteModal')).show();
}
</script>