<?php
// バックアップページ

// バックアップ実行（HTMLが出力される前に処理）
if (isset($_POST['action']) && $_POST['action'] === 'backup') {
    // エラー報告を一時的に有効化
    error_reporting(E_ALL);
    ini_set('display_errors', 1);
    
    $selectedTargets = $_POST['targets'] ?? [];
    
    if (empty($selectedTargets)) {
        $_SESSION['backup_error'] = 'バックアップ対象を選択してください。';
    } else {
        try {
            // バックアップデータを収集
            $backupData = [];
            
            // メタデータを追加
            $metadata = [
                'backup_date' => date('Y-m-d H:i:s'),
                'version' => '1.0',
                'targets' => $selectedTargets,
                'site_url' => $_SERVER['HTTP_HOST'] ?? 'unknown'
            ];
            
            // バックアップディレクトリを作成（プロジェクト内）
            $backupDir = __DIR__ . '/../../backups';
            if (!is_dir($backupDir)) {
                if (!mkdir($backupDir, 0755, true)) {
                    throw new Exception('バックアップディレクトリを作成できません');
                }
            }
            
            // JSONファイルとして保存
            $backupFile = $backupDir . '/backup_' . date('Ymd_His') . '.json';
            
            // 各ターゲットのデータを収集
            $backupTargets = [
                'articles' => [
                    'name' => '記事データ',
                    'files' => [
                        __DIR__ . '/../../data/articles.json',
                        __DIR__ . '/../../data/history.json'
                    ]
                ],
                'presets' => [
                    'name' => 'プリセット',
                    'files' => [
                        __DIR__ . '/../../data/presets.json',
                        __DIR__ . '/../../data/default-presets.json'
                    ]
                ],
                'configs' => [
                    'name' => '設定・API情報',
                    'files' => [
                        __DIR__ . '/../../config.php'
                    ]
                ],
                'site_configs' => [
                    'name' => 'サイト設定',
                    'files' => [
                        __DIR__ . '/../../data/site-configs.json'
                    ]
                ],
                'generated_sites' => [
                    'name' => '生成済みサイト',
                    'dirs' => [
                        __DIR__ . '/../../outputs'
                    ]
                ]
            ];
            
            foreach ($selectedTargets as $target) {
                if (isset($backupTargets[$target])) {
                    $backupData[$target] = [];
                    
                    // ファイルを読み込み
                    if (isset($backupTargets[$target]['files'])) {
                        foreach ($backupTargets[$target]['files'] as $file) {
                            if (file_exists($file)) {
                                // config.phpの場合は特別処理
                                if (basename($file) === 'config.php') {
                                    // 現在の設定を取得
                                    $currentConfig = require $file;
                                    
                                    // APIキーを含む実際の設定内容でPHPコードを生成
                                    $configCode = "<?php\n// 設定ファイル\n\n\$config = [\n";
                                    
                                    // APIキー
                                    $configCode .= "    // APIキー\n    'api_keys' => [\n";
                                    foreach ($currentConfig['api_keys'] as $key => $value) {
                                        $configCode .= "        '$key' => '" . addslashes($value) . "',\n";
                                    }
                                    $configCode = rtrim($configCode, ",\n") . "\n    ],\n    \n";
                                    
                                    // DMM設定
                                    $configCode .= "    // DMM設定\n    'dmm' => [\n";
                                    $configCode .= "        'api_id' => '" . addslashes($currentConfig['dmm']['api_id'] ?? '') . "',\n";
                                    $configCode .= "        'affiliate_id' => '" . addslashes($currentConfig['dmm']['affiliate_id'] ?? '') . "'\n";
                                    $configCode .= "    ],\n    \n";
                                    
                                    // DUGA設定
                                    $configCode .= "    // DUGA設定\n    'duga' => [\n";
                                    $configCode .= "        'app_id' => '" . addslashes($currentConfig['duga']['app_id'] ?? '') . "',\n";
                                    $configCode .= "        'agent_id' => '" . addslashes($currentConfig['duga']['agent_id'] ?? '') . "',\n";
                                    $configCode .= "        'banner_id' => '" . addslashes($currentConfig['duga']['banner_id'] ?? '') . "'\n";
                                    $configCode .= "    ],\n    \n";
                                    
                                    // 管理画面設定（パスワードはハッシュのまま保存）
                                    $configCode .= "    // 管理画面設定\n";
                                    $configCode .= "    'admin_password' => '" . addslashes($currentConfig['admin_password']) . "',\n";
                                    $configCode .= "    'session_timeout' => " . ($currentConfig['session_timeout'] ?? 86400) . ",\n    \n";
                                    
                                    // FTPプロファイル
                                    $configCode .= "    // FTPプロファイル\n    'ftp_profiles' => " . var_export($currentConfig['ftp_profiles'] ?? [], true);
                                    
                                    $configCode .= "\n];\n\nrequire_once __DIR__ . '/config.default.php';\n";
                                    $configCode .= "\$config = array_merge(\$defaultConfig, \$config);\n\nreturn \$config;\n?>";
                                    
                                    $backupData[$target][basename($file)] = $configCode;
                                } else {
                                    $content = file_get_contents($file);
                                    // JSONファイルの場合はデコードして保存（AIの思考過程などの不正なデータを除去）
                                    if (pathinfo($file, PATHINFO_EXTENSION) === 'json') {
                                        $jsonData = json_decode($content, true);
                                        if ($jsonData !== null) {
                                            $backupData[$target][basename($file)] = $jsonData;
                                        } else {
                                            $backupData[$target][basename($file)] = $content;
                                        }
                                    } else {
                                        $backupData[$target][basename($file)] = $content;
                                    }
                                }
                            }
                        }
                    }
                    
                    // ディレクトリは除外（JSONでは扱いにくいため）
                }
            }
            
            // メタデータを追加
            $backupData['_metadata'] = $metadata;
            
            // JSONファイルとして保存
            if (file_put_contents($backupFile, json_encode($backupData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
                // ダウンロード用のセッション変数を設定
                $_SESSION['download_backup'] = basename($backupFile);
                $_SESSION['backup_success'] = 'バックアップを作成しました。';
            } else {
                throw new Exception('バックアップファイルの作成に失敗しました。');
            }
            
        } catch (Exception $e) {
            $_SESSION['backup_error'] = 'バックアップエラー: ' . $e->getMessage();
        }
    }
    
    header('Location: ?page=backup');
    exit;
}

// ダウンロード処理
if (isset($_GET['download'])) {
    $filename = isset($_GET['file']) ? $_GET['file'] : ($_SESSION['download_backup'] ?? '');
    
    // セキュリティチェック
    if (!empty($filename) && preg_match('/^backup_[0-9]{8}_[0-9]{6}\.json$/', $filename)) {
        $filepath = __DIR__ . '/../../backups/' . $filename;
        
        if (file_exists($filepath)) {
            header('Content-Type: application/octet-stream');
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            header('Content-Length: ' . filesize($filepath));
            header('Cache-Control: no-cache, must-revalidate');
            header('Pragma: no-cache');
            readfile($filepath);
            unset($_SESSION['download_backup']);
            exit;
        }
    }
}

// バックアップ対象のファイル・データを定義
$backupTargets = [
    'articles' => [
        'name' => '記事データ',
        'files' => [
            __DIR__ . '/../../data/articles.json',
            __DIR__ . '/../../data/history.json'
        ]
    ],
    'presets' => [
        'name' => 'プリセット',
        'files' => [
            __DIR__ . '/../../data/presets.json',
            __DIR__ . '/../../data/default-presets.json'
        ]
    ],
    'configs' => [
        'name' => '設定・API情報',
        'files' => [
            __DIR__ . '/../../config.php'
        ]
    ],
    'site_configs' => [
        'name' => 'サイト設定',
        'files' => [
            __DIR__ . '/../../data/site-configs.json'
        ]
    ],
    'generated_sites' => [
        'name' => '生成済みサイト',
        'dirs' => [
            __DIR__ . '/../../outputs'
        ]
    ]
];

// バックアップ実行とダウンロード処理は上部で実行済み

// 既存のバックアップを取得
$backupsDir = __DIR__ . '/../../backups';
$existingBackups = [];
if (is_dir($backupsDir)) {
    $files = glob($backupsDir . '/*.json');
    foreach ($files as $file) {
        $existingBackups[] = [
            'filename' => basename($file),
            'size' => filesize($file),
            'date' => filemtime($file)
        ];
    }
    // 日付順にソート（新しい順）
    usort($existingBackups, function($a, $b) {
        return $b['date'] - $a['date'];
    });
}

// ヘルパー関数
// バックアップディレクトリをコピーする関数は不要になったため削除

function formatFileSize($bytes) {
    $units = ['B', 'KB', 'MB', 'GB'];
    $i = 0;
    while ($bytes >= 1024 && $i < count($units) - 1) {
        $bytes /= 1024;
        $i++;
    }
    return round($bytes, 2) . ' ' . $units[$i];
}
?>

<div class="page-header">
    <h1><i class="bi bi-archive"></i> バックアップ管理</h1>
    <p class="text-muted">重要なデータのバックアップを作成・管理します</p>
</div>

<?php if (isset($_SESSION['backup_success'])): ?>
<div class="alert alert-success alert-dismissible fade show" role="alert">
    <i class="bi bi-check-circle"></i> <?php echo $_SESSION['backup_success']; ?>
    <?php if (isset($_SESSION['download_backup'])): ?>
        <a href="../api/download-backup.php" class="alert-link">ダウンロード</a>
    <?php endif; ?>
    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
</div>
<?php
unset($_SESSION['backup_success']);
?>
<?php endif; ?>

<?php if (isset($_SESSION['backup_error'])): ?>
<div class="alert alert-danger alert-dismissible fade show" role="alert">
    <i class="bi bi-exclamation-triangle"></i> <?php echo $_SESSION['backup_error']; ?>
    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
</div>
<?php
unset($_SESSION['backup_error']);
?>
<?php endif; ?>

<div class="row">
    <div class="col-lg-6">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-plus-circle"></i> 新規バックアップ作成</h5>
            </div>
            <div class="card-body">
                <form method="POST" action="">
                    <input type="hidden" name="action" value="backup">
                    
                    <div class="mb-3">
                        <label class="form-label">バックアップ対象を選択</label>
                        
                        <?php foreach ($backupTargets as $key => $target): ?>
                        <div class="form-check">
                            <input class="form-check-input" type="checkbox" name="targets[]" 
                                   value="<?php echo $key; ?>" id="target_<?php echo $key; ?>" checked>
                            <label class="form-check-label" for="target_<?php echo $key; ?>">
                                <strong><?php echo $target['name']; ?></strong>
                                <?php if ($key === 'articles'): ?>
                                    <small class="text-muted">（記事データ、履歴）</small>
                                <?php elseif ($key === 'configs'): ?>
                                    <small class="text-muted">（APIキー、FTPプロファイルを含む）</small>
                                <?php elseif ($key === 'generated_sites'): ?>
                                    <small class="text-muted">（生成済みのHTMLファイル）</small>
                                <?php endif; ?>
                            </label>
                        </div>
                        <?php endforeach; ?>
                    </div>
                    
                    <div class="alert alert-info">
                        <i class="bi bi-info-circle"></i> 
                        バックアップにはAPIキーやFTP設定などの機密情報が含まれます。
                        安全な場所に保管してください。
                    </div>
                    
                    <button type="submit" class="btn btn-primary">
                        <i class="bi bi-download"></i> バックアップを作成
                    </button>
                </form>
            </div>
        </div>
    </div>
    
    <div class="col-lg-6">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="bi bi-clock-history"></i> バックアップ履歴</h5>
            </div>
            <div class="card-body">
                <?php if (empty($existingBackups)): ?>
                    <p class="text-muted">まだバックアップがありません。</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table table-sm">
                            <thead>
                                <tr>
                                    <th>ファイル名</th>
                                    <th>サイズ</th>
                                    <th>作成日時</th>
                                    <th>操作</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach (array_slice($existingBackups, 0, 10) as $backup): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($backup['filename']); ?></td>
                                    <td><?php echo formatFileSize($backup['size']); ?></td>
                                    <td><?php echo date('Y/m/d H:i', $backup['date']); ?></td>
                                    <td>
                                        <a href="?page=backup&download=1&file=<?php echo htmlspecialchars($backup['filename']); ?>" 
                                           class="btn btn-sm btn-outline-primary">
                                            <i class="bi bi-download"></i>
                                        </a>
                                        <button type="button" class="btn btn-sm btn-outline-danger"
                                                onclick="deleteBackup('<?php echo htmlspecialchars($backup['filename']); ?>')">
                                            <i class="bi bi-trash"></i>
                                        </button>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    
                    <?php if (count($existingBackups) > 10): ?>
                    <p class="text-muted small">
                        <i class="bi bi-info-circle"></i> 
                        最新10件を表示しています。合計: <?php echo count($existingBackups); ?>件
                    </p>
                    <?php endif; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<div class="card mt-4">
    <div class="card-header">
        <h5 class="mb-0"><i class="bi bi-upload"></i> バックアップの復元</h5>
    </div>
    <div class="card-body">
        <p>バックアップファイルから自動的にデータを復元できます。</p>
        
        <a href="../restore.php" class="btn btn-primary">
            <i class="bi bi-arrow-clockwise"></i> 復元ツールを開く
        </a>
        
        <div class="alert alert-info mt-3">
            <h6 class="alert-heading"><i class="bi bi-info-circle"></i> 復元ツールの特徴</h6>
            <ul class="mb-0">
                <li>新旧どちらのバックアップ形式にも対応</li>
                <li>復元する項目を選択可能</li>
                <li>現在のデータを自動バックアップしてから復元</li>
                <li>記事IDの重複を自動的に回避</li>
            </ul>
        </div>
        
        <div class="alert alert-warning mt-2">
            <i class="bi bi-exclamation-triangle"></i> 
            <strong>注意:</strong> 設定・API情報を復元すると、現在のAPIキーやFTP設定が上書きされます。
            必要に応じて事前にバックアップを作成してください。
        </div>
    </div>
</div>

<script>
function deleteBackup(filename) {
    if (!confirm('このバックアップを削除しますか？\n' + filename)) {
        return;
    }
    
    // 削除処理を実装
    fetch('../api/delete-backup.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({ filename: filename })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            location.reload();
        } else {
            alert('削除に失敗しました: ' + data.error);
        }
    })
    .catch(error => {
        alert('エラーが発生しました');
        console.error(error);
    });
}
</script>